<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\ApiV1;

use GuzzleHttp\Promise\EachPromise;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Exception\ItemNotFoundException;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter\ArticleAdapter;
use Psr\Log\LoggerAwareInterface;

class ArticleService extends AbstractService
{
    use LanguageAware;

    const DEFAULT_BASE_URI = 'https://mein.toubiz.de';
    /** @var string */
    protected $detailUriPath = 'api/v1/article';
    /** @var string */
    protected $listUriPath = 'api/v1/article';

    public function fetchArticles(callable $block): ServiceResult
    {
        $ids = $this->listAllIds();
        $total = count($ids);

        $pool = new EachPromise($this->requestDetailPages($ids), [
            'concurrency' => $this->parameters['concurrency'] ?? 10,
            'fulfilled' => function ($data) use ($block, $total) {
                $adapter = $this->makeAdapter($data);
                if ($adapter instanceof LoggerAwareInterface) {
                    $adapter->setLogger($this->logger);
                }
                $block($adapter, $total);
            },
        ]);

        $pool->promise()->wait();

        $result = new ServiceResult();
        $result->setAll($ids);
        return $result;
    }

    public function fetchArticle(string $id, callable $importCallable, callable $deleteCallable): ServiceResult
    {
        $query = $this->getDetailQueryParams();

        $this->requestDetailPage($query, $id)
            ->then(
                function ($data) use ($importCallable) {
                    $adapter = $this->makeAdapter($data);
                    if ($adapter instanceof LoggerAwareInterface) {
                        $adapter->setLogger($this->logger);
                    }
                    $importCallable($adapter);
                },
                function ($data) use ($deleteCallable, $id) {
                    if ($data instanceof ItemNotFoundException) {
                        $deleteCallable($id);
                    }
                }
            )
            ->wait();

        $result = new ServiceResult();
        $result->setAll([ $id ]);
        return $result;
    }

    protected function makeAdapter(array $data): ArticleAdapterInterface
    {
        return new ArticleAdapter($data, $this->parameters['type']);
    }

    protected function listAllIds(): array
    {
        $query = array_replace_recursive($this->parameters['request'] ?? [], $this->getListRequestParams());

        if ($this->delta !== null && $this->delta->asYears() < 1) {
            $query['filter'] = $query['filter'] ?? [];
            $query['filter']['updatedAfter'] = (new \DateTime())->sub($this->delta)->format('Y-m-d\TH:i:s');
        }

        if ($this->id) {
            $query['filter'] = $query['filter'] ?? [];
            $query['filter']['id'] = $this->id;
        }

        $uri = (new Uri($this->listUriPath))->withQuery(http_build_query($query));

        return $this->jsonRequest($uri, 5, function (array $data) {
            return empty($data['_errors']);
        })->then(function (array $data) {
            return array_keys($data['payload']);
        })->wait();
    }

    protected function requestDetailPages(array $ids): \Generator
    {
        $query = $this->getDetailQueryParams();

        foreach ($ids as $id) {
            yield $this->requestDetailPage($query, $id);
        }
    }

    protected function requestDetailPage(array $query, string $id): PromiseInterface
    {
        $uri = (new Uri(sprintf('%s/%s', $this->detailUriPath, $id)))->withQuery(http_build_query($query));
        return $this->jsonRequest($uri, 5, function ($data) {
            return empty($data['_errors']);
        })->then(function ($data) {
            return $data['payload'];
        });
    }

    protected function getDetailQueryParams(): array
    {
        return [
            'language' => $this->language,
            'api_token' => $this->apiKey,
            'include' => [
                'additionalAddress',
                'awards',
                'awardValues',
                'capacity',
                'channelContext',
                'classifications',
                'client',
                'contactInformation',
                'emails',
                'externalIds',
                'fieldBlueprints',
                'fieldValues',
                'files',
                'floodlights',
                'geometryForStages',
                'languages',
                'links',
                'media',
                'openingTimes',
                'partners',
                'permissions',
                'phoneNumbers',
                'relatedArticles',
                'relatedEvents',
                'seo',
                'services',
                'snowReports',
                'tourVariants',
                'tags',
            ],
            'sizes' => [
                'preview' => $this->parameters['previewSize'] ?? '600',
            ],
        ];
    }

    protected function getListRequestParams(): array
    {
        return [
            'api_token' => $this->apiKey,
            'minimal' => true,
            'language' => $this->language,
        ];
    }
}
