<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter;

use DateTimeInterface;
use Newland\Toubiz\Api\Constants\EventScope;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\EventAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\EventDateAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\EventTagAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Sync\Neos\Enum\ArticleType;

class EventAdapter extends AbstractObjectAdapter implements
    EventAdapterInterface,
    HasLocationDataInterface
{
    use LanguageAware,
        EventCommon,
        WithCitySelectors;

    protected const SCOPE_MAPPING = [
        'local' => EventScope::SCOPE_LOCAL,
        'regional' => EventScope::SCOPE_REGIONAL,
        'multiregional' => EventScope::SCOPE_TRANSREGIONAL,
    ];

    public function getDataSource(): string
    {
        return self::DATA_SOURCE_TOUBIZ_APIV1;
    }

    public function getEventType(): string
    {
        return $this->object['eventType'];
    }

    public function getName(): string
    {
        return $this->object['name'];
    }

    public function getDescription(): string
    {
        return $this->object['description'];
    }

    public function getAdditionalInformation(): string
    {
        return $this->object['intro'];
    }

    public function getBeginsAt(): ?DateTimeInterface
    {
        if ($this->object['firstDate']) {
            return $this->buildDateTimeFromStrings(
                $this->object['firstDate']['date'],
                $this->object['firstDate']['startAt']
            );
        }

        return null;
    }

    public function getEndsAt(): ?DateTimeInterface
    {
        if ($this->object['lastDate']) {
            return $this->buildDateTimeFromStrings(
                $this->object['lastDate']['date'],
                $this->object['lastDate']['endAt']
            );
        }

        return null;
    }

    public function getAdmission(): string
    {
        return $this->object['additionalBookingInformation'];
    }

    public function getIsHighlight(): bool
    {
        return $this->object['highlight'];
    }

    public function getIsTipp(): bool
    {
        return $this->object['highlight'];
    }

    public function getTicketUri(): ?string
    {
        return $this->object['bookingUrl'] ?: null;
    }

    public function getTicketContact(): ?string
    {
        return $this->object['bookingHotline'] ?: null;
    }

    public function getLink(): ?string
    {
        return ($this->object['contactInformation']['website'] ?? null) ?: null;
    }

    public function getUpdatedAt(): DateTimeInterface
    {
        return new \DateTime();
    }

    /** @return CategoryAdapterInterface[] */
    public function getCategories(): array
    {
        return [ new CategoryAdapter($this->object['category']) ];
    }

    /** @return EventDateAdapterInterface[] */
    public function getEventDates(): array
    {
        $adapters = [];

        foreach ($this->object['dates'] ?? [] as $date) {
            $adapters[] = new EventDateAdapter($date);
        }

        return $adapters;
    }

    /** @return EventTagAdapterInterface[] */
    public function getEventTags(): array
    {
        return array_map(
            function ($tag) {
                return new EventTagAdapter($tag);
            },
            $this->object['tags']
        );
    }

    public function getAttributes(): array
    {
        // TODO: Implement getAttributes() method.
        return [];
    }

    /** @return MediumAdapterInterface[] */
    public function getMedia(): array
    {
        return array_map(
            function ($media) {
                return new MediumAdapter($media);
            },
            $this->object['media']
        );
    }

    public function getOrganizer(): ?AddressAdapterInterface
    {
        return null;
    }

    public function getLocation(): ?AddressAdapterInterface
    {
        $location = ($this->object['location'] ?? $this->object['locationData']) ?? null;
        if (!$location || !$location['address']) {
            return null;
        }

        return new ArticleBasedAddressAdapter($location, $location['address']);
    }

    public function getSourceName(): ?string
    {
        return $this->object['author'];
    }

    public function getScope(): int
    {
        return static::SCOPE_MAPPING[$this->object['type']] ?? EventScope::SCOPE_UNDEFINED;
    }

    public function getAdditionalSearchString(): ?string
    {
        return $this->object['seo']['searchKeywords'] ?? null;
    }

    public function getExternalId(): string
    {
        return $this->object['id'];
    }

    public function getIsCanceled(): bool
    {
        return $this->object['canceled'];
    }

    public function getAreaName(): ?string
    {
        if ($this->object['location'] ?? null) {
            if ($this->object['location']['type'] === 'point') {
                return $this->object['location']['address']['city'] ?? null;
            }
            if ($this->object['location']['type'] === 'area') {
                return $this->object['location']['name'];
            }
            return null;
        }

        return $this->object['locationData']['address']['city'] ?? null;
    }

    public function getCanonicalUrl(): ?string
    {
        return $this->object['seo']['canonicalLink'] ?? null;
    }
}
