<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\EventApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Constants\EventScope;
use Newland\Toubiz\Api\Constants\EventType;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\Concern\MediumConstants;
use Newland\Toubiz\Api\ObjectAdapter\EventAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\EventApiService\EventDate\EventDateAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\EventApiService\EventDate\EventDateFactory;
use Newland\Toubiz\Api\Utility\UrlUtility;

/**
 * Event adapter.
 */
class EventAdapter extends AbstractLegacyObjectAdapter implements
    HasLocationDataInterface,
    EventAdapterInterface
{
    use LanguageAware;

    protected const SCOPE_MAPPING = [
        0 => EventScope::SCOPE_UNDEFINED,
        1 => EventScope::SCOPE_LOCAL,
        2 => EventScope::SCOPE_REGIONAL,
        3 => EventScope::SCOPE_TRANSREGIONAL,
    ];

    /** @var string[] */
    private $detailUriTemplates = [];

    public function setDetailUriTemplates(array $detailUriTemplates): void
    {
        $this->detailUriTemplates = $detailUriTemplates;
    }

    public function getDataSource(): string
    {
        return self::DATA_SOURCE_TOUBIZ_LEGACY;
    }

    public function getEventType(): string
    {
        return EventType::EVENT;
    }

    public function getExternalId(): string
    {
        return strip_tags((string) $this->object['id_event']);
    }

    public function getName(): string
    {
        return strip_tags((string) $this->object['name']);
    }

    public function getDescription(): string
    {
        $descriptionPart = $this->findInArray(
            $this->object['descriptions'],
            'descriptionTypeName',
            'description'
        );

        return $descriptionPart ? (string) $descriptionPart['description'] : '';
    }

    public function getAdditionalInformation(): string
    {
        $descriptionPart = $this->findInArray(
            $this->object['descriptions'],
            'descriptionTypeName',
            'event_info'
        );

        return $descriptionPart ? (string) $descriptionPart['description'] : '';
    }

    public function getAdmission(): string
    {
        $descriptionPart = $this->findInArray(
            $this->object['descriptions'],
            'descriptionTypeName',
            'price'
        );

        return $descriptionPart ? (string) $descriptionPart['description'] : '';
    }

    public function getBeginsAt(): \DateTimeInterface
    {
        return $this->buildDateTimeFromStrings(
            $this->object['date_from'],
            $this->object['time_from']
        );
    }

    public function getEndsAt(): \DateTimeInterface
    {
        $startDate = $this->buildDateTimeFromStrings(
            $this->object['date_from'],
            $this->object['time_from']
        );
        $endDate = $this->buildDateTimeFromStrings(
            $this->object['date_to'],
            $this->object['time_to']
        );
        if ($startDate->format('Y-m-d H:i:s') === $endDate->format('Y-m-d H:i:s')) {
            $endDate->modify('+1 day');
        }
        if ($startDate->format('Y-m-d H:i:s') > $endDate->format('Y-m-d H:i:s')) {
            $endDate->modify('+1 day');
        }
        return $endDate;
    }

    public function getIsHighlight(): bool
    {
        return (bool) $this->object['is_highlight'];
    }

    public function getIsTipp(): bool
    {
        return (bool) $this->object['is_tipp'];
    }

    public function getUpdatedAt(): \DateTimeInterface
    {
        return new \DateTime($this->object['timestamp']);
    }

    public function getTicketUri(): ?string
    {
        return $this->parseFromDeeplink('event_deeplink', 'deeplink');
    }

    public function getTicketContact(): ?string
    {
        $ticketContact = $this->parseFromDeeplink('event_deeplink', 'deeplinkContact');

        if (empty($ticketContact)) {
            $ticketContact = $this->parseFromDeeplink('event_deeplink_contact', 'deeplinkContact');
        }

        return $ticketContact;
    }

    public function getLink(): ?string
    {
        if (!empty($this->object['link']) && $this->object['link'] !== 'http://') {
            return (string) $this->object['link'];
        }
        return null;
    }

    /** @return CategoryAdapter[] */
    public function getCategories(): array
    {
        return array_map(
            function ($item) {
                return (new CategoryAdapter($item));
            },
            (array) $this->object['category']
        );
    }

    /** @return EventTagAdapter[] */
    public function getEventTags(): array
    {
        return array_map(
            function ($item) {
                return (new EventTagAdapter($item));
            },
            (array) $this->object['keywords']
        );
    }

    public function getAttributes(): array
    {
        $features = [];
        foreach ((array) $this->object['features'] as $feature) {
            $features[] = strip_tags((string) $feature['feature']);
        }

        return [ 'features' => $features ];
    }

    /** @return EventDateAdapter[] */
    public function getEventDates(): array
    {
        return array_map(
            function ($item) {
                $item['id_event_extern'] = $this->object['id_event_extern'] ?? null;
                return EventDateFactory::get(
                    $item,
                    $this->object['channel_id'] ?? null,
                    $this->detailUriTemplates
                );
            },
            (array) $this->object['dates']
        );
    }

    /** @return MediumAdapter[] */
    public function getMedia(): array
    {
        $media = [];

        foreach ((array) ($this->object['multimedia'] ?? []) as $item) {
            $adapter = new MediumAdapter($item);
            if ($adapter->getType() === MediumConstants::TYPE_IMAGE
                && UrlUtility::isValidUrl($adapter->getSourceUri())
            ) {
                $media[] = $adapter;
            }
        }
        return $media;
    }

    public function getOrganizer(): ?AddressAdapterInterface
    {
        $item = $this->findInArray(
            $this->object['address'],
            'addressTypeName',
            'organizer'
        );

        // Import without an ID is not possible.
        if ($item && $item['addressTypeID']) {
            return new AddressAdapter($item);
        }

        return null;
    }

    public function getLocation(): ?AddressAdapterInterface
    {
        $item = $this->findInArray(
            $this->object['address'],
            'addressTypeName',
            'location'
        );

        $eventLocation = $this->findInArray(
            $this->object['location'] ?? [],
            'locationTypeName',
            'event_location'
        );
        $nameFallback = $eventLocation['location'] ?? null;

        // Import without an ID is not possible.
        if ($item && $item['addressTypeID']) {
            return new AddressAdapter($item, $nameFallback);
        }

        return null;
    }

    public function getAreaName(): ?string
    {
        return null;
    }

    public function getSourceName(): ?string
    {
        $source = $this->object['quelle'] ?? null;
        return $source ? strip_tags((string) $source) : null;
    }

    public function getScope(): int
    {
        $meaning = $this->object['meaning'] ?? null;
        return self::SCOPE_MAPPING[$meaning] ?? EventScope::SCOPE_UNDEFINED;
    }

    public function parseFromDeeplink(string $deeplinkType, string $field): ?string
    {
        $value = null;

        $part = $this->findInArray(
            $this->object['deeplinks'],
            'deeplinkTypeName',
            $deeplinkType
        );

        if (is_array($part)) {
            $value = $part[$field] ?? null;
        }

        return $value ? strip_tags((string) $value) : null;
    }

    public function getCitySelectors(): array
    {
        $selectors = [];

        foreach ($this->object['location'] ?? [] as $location) {
            if (array_key_exists('locationTypeName', $location) &&
                $location['locationTypeName'] === 'city') {
                $selectors[] = new ExternalIdSelector(ExternalIdType::TOUBIZ_LEGACY, $location['locationTypeID']);
            }
        }

        return $selectors;
    }

    public function getAdditionalSearchString(): ?string
    {
        $searchWords = array_map(
            function ($item) {
                return $item['keyword'] ?? '';
            },
            $this->object['searchwords'] ?? []
        );

        $additionalSearchString = implode(' ', $searchWords);
        return $additionalSearchString ?: null;
    }

    public function getIsCanceled(): bool
    {
        return false;
    }
}
