<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\MonthConstants;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\FileAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\WithCacheProperty;

/**
 * Tour adapter.
 *
 * Describes a tour.
 */
class TourAdapter extends AbstractObjectAdapter implements
    ArticleAdapterInterface,
    HasLocationDataInterface
{
    use LanguageAware;

    /** @var array|null */
    private $cachedGeometry;

    public function getSourceSystem(): string
    {
        return self::SOURCE_OUTDOORACTIVE;
    }

    public function getSourceApiVersion(): ?string
    {
        return null;
    }

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE . '_' . $this->object['id'];
    }

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_TOUR;
    }

    public function getName(): string
    {
        $name = $this->object['title'] ?? null;
        return strip_tags((string) $name);
    }

    public function getAbstract(): ?string
    {
        $value = $this->get('shortText');
        return $value ? $this->stringCleaner->purifyHtml((string) $value) : null;
    }

    public function getDescription(): ?string
    {
        $value = $this->get('longText');
        return $value ? $this->stringCleaner->purifyHtml((string) $value) : null;
    }

    public function getLatitude(): ?float
    {
        $geometry = $this->extractGeometry();
        if (\count($geometry) > 0) {
            return $geometry[0][0];
        }

        return null;
    }

    public function getLongitude(): ?float
    {
        $geometry = $this->extractGeometry();
        if (\count($geometry) > 0) {
            return $geometry[0][1];
        }

        return null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        $geometry = $this->extractGeometry();
        $object = $this->object;
        if (\count($geometry) > 0) {
            $object['firstGeometry'] = [
                'lat' => $geometry[0][0],
                'lon' => $geometry[0][1],
            ];
        }
        return new AddressAdapter($object);
    }

    public function getMainCategory(): ?CategoryAdapterInterface
    {
        $categories = $this->getCategories();

        return $categories[0] ?? null;
    }

    /**
     * @return CategoryAdapterInterface[]
     */
    public function getCategories(): array
    {
        $category = $this->get('category');
        return $category ? [ new CategoryAdapter($category) ] : [];
    }

    /**
     * @return MediumAdapterInterface[]
     */
    public function getMedia(): array
    {
        $images = $this->get('images.image', []);
        usort(
            $images,
            function ($imageA, $imageB) {
                return ($imageA['id'] === $this->get('primaryImage.id')) ? -1 : 1;
            }
        );

        return array_map(
            function ($data) {
                return new MediumAdapter($data);
            },
            $images
        );
    }

    public function getMainMedium(): ?MediumAdapterInterface
    {
        foreach ($this->get('images.image', []) as $image) {
            if ($this->get('primaryImage.id') === $image['id']) {
                return new MediumAdapter($image);
            }
        }

        $primaryImage = $this->get('images.image.0');
        return $primaryImage ? new MediumAdapter($primaryImage) : null;
    }

    /**
     * @return FileAdapterInterface[]
     */
    public function getFiles(): array
    {
        return [];
    }

    /**
     * @return bool
     */
    public function hasAttributes(): bool
    {
        return true;
    }

    /**
     * @return array
     */
    public function getAttributes(): array
    {
        $items = [
            new AttributeAdapter($this->getExternalId(), 'geometry', $this->extractGeometry()),
        ];

        foreach ($this->get('properties.property', []) as $property) {
            $items[] = new AttributeAdapter(
                $this->getExternalId(),
                'properties',
                (string) $property['tag']
            );
        }

        foreach ($this->get('labels', []) as $attribute => $value) {
            $items[] = new AttributeAdapter($this->getExternalId(), 'labels', $attribute);
        }


        foreach ($this->get('crossCountryTechnique.property', []) as $technique) {
            $items[] = new AttributeAdapter(
                $this->getExternalId(),
                'crossCountryTechnique',
                (string) $technique['tag']
            );
        }


        $months = [
            'jan' => MonthConstants::JANUARY,
            'feb' => MonthConstants::FEBRUARY,
            'mar' => MonthConstants::MARCH,
            'apr' => MonthConstants::APRIL,
            'may' => MonthConstants::MAY,
            'jun' => MonthConstants::JUNE,
            'jul' => MonthConstants::JULY,
            'aug' => MonthConstants::AUGUST,
            'sep' => MonthConstants::SEPTEMBER,
            'oct' => MonthConstants::OCTOBER,
            'nov' => MonthConstants::NOVEMBER,
            'dec' => MonthConstants::DECEMBER,
        ];

        foreach ($months as $external => $internal) {
            if ($this->get(sprintf('season.%s', $external))) {
                $items[] = new AttributeAdapter($this->getExternalId(), 'bestSeason', $internal);
            }
        }

        // Outdooractive uses a 1-3 scale as opposed to the
        // 0-2 scale used by toubiz
        // difficulty 5 is now 'unknown'
        $rawValue = (int) ($this->get('rating.difficulty', 3));
        $difficulty = max($rawValue - 1, 0);

        $simpleAttributes = [
            TourAttributes::ASCEND_ELEVATION => (int) ($this->get('elevation.ascent')),
            TourAttributes::DESCENT_ELEVATION => (int) ($this->get('elevation.descent')),
            TourAttributes::CONDITION_RATING => (int) $this->get('rating.condition'),
            TourAttributes::DIFFICULTY_RATING => $difficulty,
            TourAttributes::NECESARRY_EQUIPMENT =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('equipment')),
            TourAttributes::EXPERIENCE_RATING => (int) $this->get('rating.qualityOfExperience'),
            TourAttributes::LANDSCAPE_RATING => (int) $this->get('rating.landscape'),
            TourAttributes::MAXIMUM_ALTITUDE => (int) $this->get('elevation.maxAltitude'),
            TourAttributes::MINIMUM_ALTITUDE => (int) $this->get('elevation.minAltitude'),
            TourAttributes::PUBLIC_TRANSIT =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('publicTransit')),
            TourAttributes::PRIVATE_TRANSIT =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('gettingThere')),
            TourAttributes::STARTING_POINT =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('startingPointDescr')),
            TourAttributes::ENDING_POINT =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('tour.destination')),
            TourAttributes::TECHNIQUE_RATING => (int) $this->get('rating.technique'),
            TourAttributes::TOUR_DURATION => (int) $this->get('time.min'),
            TourAttributes::TOUR_LENGTH => (int) ($this->get('length')),
            TourAttributes::DATA_SOURCE => TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE,
            TourAttributes::DATA_SOURCE_ID =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('id')),
            TourAttributes::DIRECTIONS => $this->stringCleaner->cleanHtmlString((string) $this->get('directions')),
            TourAttributes::PARKING => $this->stringCleaner->cleanHtmlString((string) $this->get('parking')),
            TourAttributes::ADDITIONAL_INFORMATION =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('additionalInformation')),
            TourAttributes::HINT_OF_AUTHOR => $this->stringCleaner->cleanHtmlString((string) $this->get('tip')),
            TourAttributes::SAFETY_NOTE =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('safetyGuidelines')),
            TourAttributes::LITERATURE_TIP_OF_AUTHOR =>
                $this->stringCleaner->cleanHtmlString((string) $this->get('literature')),
        ];

        foreach ($simpleAttributes as $name => $data) {
            if (!is_string($data) || $data !== '') {
                $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
            }
        }

        $opened = $this->get('opened');
        if ($opened === false || $opened === 'false') {
            $items[] = new AttributeAdapter($this->getExternalId(), TourAttributes::STATUS, 'closed');
        }

        $sourceName = $this->get('meta.source.name');
        $sourceName = $sourceName ? (string) $sourceName : null;
        if ($sourceName) {
            $items[] = new AttributeAdapter($this->getExternalId(), GenericArticleAttributes::CLIENT_NAME, $sourceName);
        }

        return $items;
    }


    /** @return float[][] */
    private function extractGeometry(): array
    {
        if ($this->cachedGeometry) {
            return $this->cachedGeometry;
        }

        $rawGeometry = $this->get('geometry');
        $geometry = [];
        if ($rawGeometry) {
            $geoItems = explode(' ', $rawGeometry);
            foreach ($geoItems as $geoItem) {
                $geo = explode(',', $geoItem);
                $geometry[] = [ (float) $geo[1], (float) $geo[0] ];
            }
        }

        $this->cachedGeometry = $geometry;
        return $geometry;
    }

    public function getBookingUris(): array
    {
        return [];
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getCitySelectors(): array
    {
        $selectors = [];

        foreach ($this->get('regions.region', []) as $region) {
            if (in_array(($region['type'] ?? null), [ 'commune', 'district' ])
                && array_key_exists('id', $region)
            ) {
                $selectors[] = new ExternalIdSelector(ExternalIdType::OUTDOORACTIVE, $region['id']);
            }
        }

        return $selectors;
    }

    public function getAdditionalSearchString(): ?string
    {
        return null;
    }
}
