<?php
namespace Newland\Toubiz\Api\Service\Tportal;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Exception\InvalidServiceResponseException;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\GuideAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\OfferAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\PackageAdapter;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use function Safe\json_decode;

/**
 * Service for legacy Toubiz TPortal.
 *
 * Concrete implementation to communicate with the TPortal
 * which is providing data for TOMAS-bound entities.
 */
class ApiService extends AbstractService
{
    use LanguageAware;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://tportal.toubiz.de';

    /**
     * Template used for detail URLs.
     * This template may contain the placeholder `{tportalUriSegment}` which will be replaced
     * with the uri segment provided by tportal.
     *
     * @var string|null
     */
    private $detailUriTemplate;

    /**
     * @var array
     */
    private $starRatingsMap = [];

    private function setStarRatingsMap(): void
    {
        $starClassificationJson = file_get_contents(
            __DIR__ . '/../../../Resources/Service/Tportal/ApiService/star-rating-map.json'
        );

        if ($starClassificationJson) {
            $this->starRatingsMap = (array) json_decode($starClassificationJson);
        }
    }

    public function fetchOffers(callable $block): ServiceResult
    {
        $concurrency = $this->parameters['concurrency'] ?? 10;
        $ids = [];

        $pool = new ConcurrentPaginatedRequests(
            $concurrency,
            function (int $page) use ($block, &$ids) {
                return $this->offerRequest($page)->then(
                    function ($data) use ($block, &$ids) {
                        // If there is only one result then `offerData` is not an array of
                        // data objects but the data object itself.
                        if (ArrayUtility::isAssociative($data['offerData'])) {
                            $data['offerData'] = [ $data['offerData'] ];
                        }

                        foreach ($data['offerData'] as $offer) {
                            $adapter = $this->offerDataToAdapter($offer, $data);
                            $ids[] = $adapter->getExternalId();
                            $block($adapter, $data['paging_txt']['count'] ?? null);
                        }
                    }
                );
            }
        );

        $pool->start()->wait();

        $result = new ServiceResult();
        $result->setAll($ids);
        return $result;
    }

    public function fetchPackages(callable $block): ServiceResult
    {
        $concurrency = $this->parameters['concurrency'] ?? 10;
        $ids = [];

        $pool = new ConcurrentPaginatedRequests(
            $concurrency,
            function (int $page) use ($block, &$ids) {
                return $this->packageRequest($page)->then(
                    function ($data) use ($block, &$ids) {
                        foreach ($data['PackageAccommodationInfos'] as $package) {
                            $adapter = $this->packageDataToAdapter($package, $data);
                            $ids[] = $adapter->getExternalId();
                            $block($adapter, $data['paging_txt']['count'] ?? null);
                        }
                    }
                );
            }
        );

        $pool->promise()->wait();

        $result = new ServiceResult();
        $result->setAll($ids);
        return $result;
    }


    public function fetchGuides(callable $block): ServiceResult
    {
        $concurrency = $this->parameters['concurrency'] ?? 10;
        $ids = [];

        $pool = new ConcurrentPaginatedRequests(
            $concurrency,
            function (int $page) use ($block, &$ids) {
                return $this->guideRequest($page)->then(
                    function ($data) use ($block, &$ids) {
                        foreach ($data['packageGuideInfos'] as $guide) {
                            $adapter = $this->guideDataToAdapter($guide, $data);
                            $ids[] = $adapter->getExternalId();
                            $block($adapter, $data['paging_txt']['count'] ?? null);
                        }
                    }
                );
            }
        );

        $pool->promise()->wait();

        $result = new ServiceResult();
        $result->setAll($ids);
        return $result;
    }

    private function offerDataToAdapter(array $offerData, array $data): OfferAdapter
    {
        $id = $offerData['serviceID'];
        $offer = new OfferAdapter(
            [
                'offer' => $offerData,
                'serviceData' => $data['serviceData'][$id],
                'availability' => $data['ServiceAvailabilities'][$id] ?? [],
                'geoResult' => $this->extractGeoResult($data['geoResultData'] ?? [], $id),
            ]
        );
        if ($this->language) {
            $offer->setLanguage($this->language);
        }
        $offer->setDetailUriTemplate($this->detailUriTemplate);
        return $offer;
    }

    private function packageDataToAdapter(array $packageData, array $data): PackageAdapter
    {
        $id = $packageData['packageID'];
        $package = new PackageAdapter(
            [
                'package' => $packageData,
                'serviceData' => $data['serviceData'][$id],
            ]
        );
        if ($this->language) {
            $package->setLanguage($this->language);
        }
        $package->setDetailUriTemplate($this->detailUriTemplate);
        return $package;
    }

    private function guideDataToAdapter(array $guideData, array $data): GuideAdapter
    {
        $id = $guideData['packageID'];
        $guide = new GuideAdapter(
            [
                'guide' => $guideData,
                'serviceData' => $data['serviceData'][$id],
                'client' => $data['configName'],
            ]
        );
        if ($this->language) {
            $guide->setLanguage($this->language);
        }
        $guide->setDetailUriTemplate($this->detailUriTemplate);
        return $guide;
    }

    /**
     * @param array $geoResultData
     * @param string $id
     * @return array|null
     */
    private function extractGeoResult(array $geoResultData, string $id)
    {
        foreach ($geoResultData as $data) {
            if (($data['serviceID'] ?? null) === $id) {
                return $data;
            }
        }
        return null;
    }

    public function setDetailUriTemplate(?string $detailUriTemplate): void
    {
        $this->detailUriTemplate = $detailUriTemplate;
    }

    private function offerRequest(int $page, int $retriesIfFail = 5): PromiseInterface
    {
        $url = $this->urlTemplate(
            '/:clientName/offer?reset=1&json=1&page=:page&offer_result_order=4',
            [
                ':clientName' => $this->clientName,
                ':page' => $page,
            ]
        );
        $url = $this->addLanguageToUrl($url);

        return $this->jsonRequest(
            new Uri($url),
            $retriesIfFail,
            function ($data) {
                if (($data['paging_txt']['count'] ?? null) === 0) {
                    throw new InvalidServiceResponseException('TPortal responded with empty response');
                }
                return $data && \count($data['offerData'] ?? []) > 0;
            }
        );
    }

    private function packageRequest(int $page, int $retriesIfFail = 5): PromiseInterface
    {
        $url = $this->urlTemplate(
            '/:clientName/package?reset=1&json=1&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );
        $url = $this->addLanguageToUrl($url);

        return $this->jsonRequest(
            new Uri($url),
            $retriesIfFail,
            function ($data) {
                if (($data['paging_txt']['count'] ?? null) === 0) {
                    throw new InvalidServiceResponseException('TPortal responded with empty response');
                }
                return \count($data['PackageAccommodationInfos'] ?? []) > 0;
            }
        );
    }

    private function guideRequest(int $page, int $retriesIfFail = 5): PromiseInterface
    {
        $url = $this->urlTemplate(
            '/:clientName/guide?reset=1&json=1&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );
        $url = $this->addLanguageToUrl($url);

        return $this->jsonRequest(
            new Uri($url),
            $retriesIfFail,
            function ($data) use ($page) {
                if (($data['paging_txt']['count'] ?? null) === 0) {
                    throw new InvalidServiceResponseException('TPortal responded with empty response');
                }
                return \count($data['packageGuideInfos'] ?? []) > 0 && $data['paging_txt']['pages'] >= $page;
            }
        );
    }

    private function urlTemplate(string $url, array $replacements): string
    {
        return str_replace(
            array_keys($replacements),
            array_map('urlencode', array_values($replacements)),
            $url
        );
    }

    private function addLanguageToUrl(string $url): string
    {
        if (!$this->language) {
            return $url;
        }

        return $url . '&lang=' . $this->language;
    }
}
