<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter;

use Newland\GpsFileParsing\Helper\XmlFileReader;
use Newland\GpsFileParsing\Model\Point;
use Newland\GpsFileParsing\ParserFactory;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\MonthConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\PointOfInterestAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\ExternalIdAdapter;
use Newland\Toubiz\Api\ObjectAdapter\HasAdditionalExternalIds;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Psr\Log\LoggerAwareTrait;

class ArticleAdapter extends AbstractObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface,
    HasAdditionalExternalIds
{
    use LanguageAware, LoggerAwareTrait;

    /** @var int */
    protected $type;

    public function __construct(array $adaptee, int $type)
    {
        $this->type = $type;
        parent::__construct($adaptee);
    }


    public function getMainType(): int
    {
        return $this->type;
    }

    public function getName(): string
    {
        return $this->object['name'];
    }

    public function getAbstract(): ?string
    {
        return $this->object['abstract'];
    }

    public function getDescription(): ?string
    {
        return $this->object['description'];
    }

    public function getLatitude(): ?float
    {
        return $this->object['point']['latitude']
            ?? $this->object['area']['latitude']
            ?? null;
    }

    public function getLongitude(): ?float
    {
        return $this->object['point']['longitude']
            ?? $this->object['area']['longitude']
            ?? null;
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        $address = $this->object['point']['address']
            ?? $this->object['area']['address']
            ?? null;

        if ($address) {
            $article = $this->object;
            $article['mainAddress'] = $address;
            return new MainAddressAdapter($article);
        }

        return null;
    }

    public function getAddresses(): array
    {
        $address = $this->getMainAddress();
        return $address ? [ $address ] : [];
    }

    public function getCategories(): array
    {
        // Indexing by id to prevent duplicate categories in primary & secondary
        $categories = [
            $this->object['primaryCategory']['id'] => $this->object['primaryCategory'],
        ];

        foreach ($this->object['secondaryCategories'] ?? [] as $item) {
            $categories[$item['id']] = $item;
        }

        return array_map(function ($item) {
            return new CategoryAdapter($item);
        }, array_values($categories));
    }

    public function getMedia(): array
    {
        return array_map(
            function ($item) {
                return new MediumAdapter($item);
            },
            $this->object['media'] ?? []
        );
    }

    public function getMainMedium(): ?MediumAdapterInterface
    {
        $medium = $this->object['media'][0] ?? null;
        if (!$medium) {
            return null;
        }
        return new MediumAdapter($medium);
    }

    public function getFiles(): array
    {
        return array_map(
            function ($item) {
                return new FileAdapter($item);
            },
            $this->object['files'] ?? []
        );
    }

    public function hasAttributes(): bool
    {
        return !empty($this->extractAttributes());
    }

    public function getAttributes(): array
    {
        return $this->extractAttributes();
    }

    public function getSourceName(): ?string
    {
        return $this->object['client']['name'] ?? null;
    }

    public function getAuthorName(): ?string
    {
        return $this->object['author'] ?? null;
    }

    public function getBookingUris(): array
    {
        return [];
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOpeningTimes(): ?string
    {
        $openingTimes = $this->object['openingTimes'] ?? null;
        if ($openingTimes) {
            return json_encode($openingTimes);
        }
        return null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        $openingTimes = $this->object['openingTimes'] ?? null;
        return $openingTimes ? 'toubizV1' : null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getExternalId(): string
    {
        return $this->object['id'];
    }

    private function extractAttributes(): array
    {
        $blueprints = [];
        $blueprintsToGroups = [];
        foreach ($this->object['fieldBlueprints'] as $group) {
            foreach ($group['fieldSets'] as $fieldSet) {
                foreach ($fieldSet['fields'] as $field) {
                    $blueprints[$field['id']] = $field;
                    $blueprintsToGroups[$field['id']] = $group['id'];
                }
            }
        }

        [ $mappedDynamicFields, $mappedFields ] = $this->extractMappedDynamicFields($blueprints, $blueprintsToGroups);
        $unmappedDynamicFields = $this->extractUnmappedAttributes($blueprints, $blueprintsToGroups, $mappedFields);
        return array_merge(
            $this->getFixed(),
            $this->getMappedRegular(),
            $mappedDynamicFields,
            $unmappedDynamicFields
        );
    }

    private function getFixed(): array
    {
        $attributes = [];

        $gpsTrackUrl = $this->object['tour']['gpsTracks'][0]['file']['url'] ?? null;
        if (is_string($gpsTrackUrl)) {
            $track = (new ParserFactory(new XmlFileReader()))->resolveParser($gpsTrackUrl)->extractTrack($gpsTrackUrl);
            if ($track->isEmpty()) {
                $this->logger->warning(sprintf(
                    'GPS File %s does not contain any track/point information.',
                    $gpsTrackUrl
                ));
            }

            $points = $track->mapPoints(function (Point $point) {
                return [ $point->getLatitude(), $point->getLongitude() ];
            });
            $attributes[] = new AttributeAdapter(
                TourAttributes::GEOMETRY,
                $points,
                null,
                $this->getExternalId() . '__geometry'
            );
        }

        $currencies = $this->object['point']['price']['currencies'] ?? [];
        $priceGroups = $this->object['point']['price']['priceGroups'] ?? [];
        if (!empty($priceGroups)) {
            $table = [];
            foreach ($priceGroups as $priceGroup) {
                foreach ($priceGroup['priceEntries'] ?? [] as $priceEntry) {
                    $row = [ $priceGroup['title'], $priceEntry['title'] ];
                    foreach ($currencies as $currency) {
                        if ($currency === 'eur') {
                            $row[] = sprintf('%.2f €', $priceEntry['eur']);
                        } elseif ($currency === 'chf') {
                            $row[] = sprintf('CHF %.2f', $priceEntry['eur']);
                        } else {
                            $row[] = sprintf('%s %.2f', $currency, $priceEntry[$currency]);
                        }
                    }
                    $row[] = $priceEntry['comment'];
                    $table[] = $row;
                }
            }

            $attributes[] = new AttributeAdapter(
                PointOfInterestAttributes::PRICES,
                $table,
                null,
                $this->getExternalId() . '__prices'
            );
        }

        $recommendedTimeOfTravel = $this->object['tour']['recommendedTimeOfTravel'] ?? null;
        if ($recommendedTimeOfTravel) {
            foreach ($recommendedTimeOfTravel as $month => $recommended) {
                if (!$recommended) {
                    continue;
                }

                $attributes[] = new AttributeAdapter(
                    TourAttributes::BEST_SEASON,
                    $this->monthToConstant($month),
                    null,
                    $this->getExternalId() . '__best-season__' . $month
                );
            }
        }

        return $attributes;
    }

    private function monthToConstant(string $month): ?string
    {
        switch ($month) {
            case 'january':
                return MonthConstants::JANUARY;
            case 'february':
                return MonthConstants::FEBRUARY;
            case 'march':
                return MonthConstants::MARCH;
            case 'april':
                return MonthConstants::APRIL;
            case 'may':
                return MonthConstants::MAY;
            case 'june':
                return MonthConstants::JUNE;
            case 'july':
                return MonthConstants::JULY;
            case 'august':
                return MonthConstants::AUGUST;
            case 'september':
                return MonthConstants::SEPTEMBER;
            case 'october':
                return MonthConstants::OCTOBER;
            case 'november':
                return MonthConstants::NOVEMBER;
            case 'december':
                return MonthConstants::DECEMBER;
            default:
                return null;
        }
    }

    private function getMappedRegular(): array
    {
        $attributes = [];
        foreach (AttributeMap::get() as [ $attributeName, $type, $path ]) {
            $attributeName = trim($attributeName);
            if ($type !== 'regular' || empty($attributeName)) {
                continue;
            }

            $value = ArrayUtility::arrayGet($this->object, explode('.', $path));
            if (is_string($value) || is_int($value) || is_bool($value)) {
                $attributes[] = new AttributeAdapter(
                    $attributeName,
                    $value,
                    null,
                    $this->getExternalId() . '__' . $attributeName
                );
            } elseif (is_array($value)) {
                foreach ($value as $index => $v) {
                    $attributes[] = new AttributeAdapter(
                        $attributeName,
                        $v,
                        null,
                        $this->getExternalId() . '__' . $attributeName . '__' . $index
                    );
                }
            }
        }
        return $attributes;
    }

    private function extractMappedDynamicFields(array $blueprints, array $blueprintsToGroups): array
    {
        $attributes = [];
        $fieldIds = [];

        foreach (AttributeMap::get() as [ $attributeName, $type, $fieldId, $fieldValue, $attributeValue ]) {
            $attributeName = trim($attributeName);
            if (empty($attributeName)
                || strpos($attributeName, '#') === 0
                || !array_key_exists($fieldId, $blueprints)
                || $type !== 'dynamicField'
            ) {
                continue;
            }

            $i = 0;
            foreach ($this->object['fieldValues'][$fieldId] ?? [] as $dynamicFieldValue) {
                foreach (AttributeAdapter::extractValueSets($dynamicFieldValue) as $value => $humanReadableValue) {
                    if ($fieldValue && $value === $fieldValue) {
                        continue;
                    }

                    $humanReadableValue = $attributeValue ?: $humanReadableValue;
                    if (strtolower($humanReadableValue) === 'true') {
                        $humanReadableValue = true;
                    } elseif (strtolower($humanReadableValue) === 'false') {
                        $humanReadableValue = false;
                    }

                    $fieldIds[] = $fieldId;
                    $attributes[] = new AttributeAdapter(
                        $attributeName,
                        $humanReadableValue,
                        $blueprintsToGroups[$fieldId],
                        sprintf('%s_%d', $blueprints[$fieldId]['id'], $i++)
                    );
                }
            }
        }

        return [ $attributes, array_unique($fieldIds) ];
    }

    private function extractUnmappedAttributes(array $blueprints, array $blueprintsToGroups, array $ignoreFields): array
    {
        $attributes = [];
        foreach ($blueprints as $id => $blueprint) {
            $i = 0;
            if (in_array($id, $ignoreFields, true) || !array_key_exists($id, $this->object['fieldValues'])) {
                continue;
            }

            foreach ($this->object['fieldValues'][$id] as $dynamicFieldValue) {
                foreach (AttributeAdapter::extractValueSets($dynamicFieldValue) as $value => $humanReadableValue) {
                    $attributes[] = new AttributeAdapter(
                        $blueprint['name'],
                        $humanReadableValue,
                        $blueprintsToGroups[$id],
                        sprintf('%s_%d', $id, $i++)
                    );
                }
            }
        }
        return $attributes;
    }

    /**
     * @inheritDoc
     */
    public function getCitySelectors(): array
    {
        return array_map(
            function (array $item) {
                return new ExternalIdSelector(ExternalIdType::TOUBIZ, $item['id']);
            },
            $this->object['relatedArticles']['areas'] ?? []
        );
    }

    /**
     * @inheritDoc
     */
    public function getAdditionalExternalIds(): array
    {
        $ids = [
            new ExternalIdAdapter(ExternalIdType::TOUBIZ, $this->getExternalId())
        ];

        foreach ($this->object['externalIds'] ?? [] as $service => $id) {
            $ids[] = new ExternalIdAdapter($service, $id);
        }

        return $ids;
    }
}
