<?php declare(strict_types=1);

namespace Newland\ToubizApi\Tests\Unit\Service\Tportal;

use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\OfferAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\PackageAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ServiceAdapterInterface;
use Newland\Toubiz\Api\Service\Tportal\ApiService;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;

class ApiServiceTest extends TestCase
{

    /** @var ApiService */
    private $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = new ApiService();
        $this->subject->setLogger(new NullLogger());
        $this->subject->setClientName('foobar');
        $this->subject->setParameters([ 'concurrency' => 1, 'sleepSecondsOnError' => 0.0 ]);
    }

    public function testFetchesOffersFromTportal(): void
    {
        $history = [];
        $this->mockNextResponses(
            [
                new Response(200, [], file_get_contents(__DIR__ . '/../../../Fixtures/TportalData/offerData.json')),
                new Response(200, [], json_encode([ 'offerData' => [] ]))
            ],
            $history
        );

        $this->subject->fetchOffers(function() {});

        $this->assertCount(2, $history, 'Api Client should have made 2 requests');
        $this->assertRegExp(
            '/^https:\/\/tportal\.toubiz\.de\/foobar\/offer/',
            (string) $history[0]['request']->getUri()
        );
        $this->assertRegExp(
            '/^https:\/\/tportal\.toubiz\.de\/foobar\/offer/',
            (string) $history[1]['request']->getUri()
        );
        $this->assertContains('page=1', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('page=2', $history[1]['request']->getUri()->getQuery());
        $this->assertContains('json=1', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('json=1', $history[1]['request']->getUri()->getQuery());
        $this->assertContains('reset=1', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('reset=1', $history[1]['request']->getUri()->getQuery());
    }


    public function testCreatesAdaptersForOffers(): void
    {
        $this->mockNextResponses(
            [
                new Response(200, [], file_get_contents(__DIR__ . '/../../../Fixtures/TportalData/offerData.json')),
                new Response(200, [], json_encode([ 'offerData' => [] ]))
            ]
        );

        $adapters = [];
        $this->subject->fetchOffers(function($adapter) use (&$adapters) {
            $adapters[] = $adapter;
            $this->assertInstanceOf(ServiceAdapterInterface::class, $adapter);
        });

        $this->assertGreaterThan(0, \count($adapters), 'The callback should have been called more than once');
    }

    public function testAddsLanguageParameterToOfferRequestIfAvailable(): void
    {
        $history = [];
        $this->mockNextResponses(
            [
                new Response(200, [], file_get_contents(__DIR__ . '/../../../Fixtures/TportalData/offerData.json')),
                new Response(200, [], json_encode([ 'offerData' => [] ]))
            ],
            $history
        );

        $this->subject->setLanguage('fr');
        $this->subject->fetchOffers(function() {});

        $this->assertCount(2, $history, 'Api Client should have made 2 requests');
        $this->assertContains('lang=fr', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('lang=fr', $history[1]['request']->getUri()->getQuery());
    }

    public function testFetchesPackagesFromTportal(): void
    {
        $history = [];
        $this->mockNextResponses(
            [
                new Response(200, [], file_get_contents(__DIR__ . '/../../../Fixtures/TportalData/packageData.json')),
                new Response(200, [], json_encode([ 'PackageAccommodationInfos' => [] ]))
            ],
            $history
        );

        $this->subject->fetchPackages(function() {});

        $this->assertCount(2, $history, 'Api Client should have made 2 requests');
        $this->assertRegExp(
            '/^https:\/\/tportal\.toubiz\.de\/foobar\/package/',
            (string) $history[0]['request']->getUri()
        );
        $this->assertRegExp(
            '/^https:\/\/tportal\.toubiz\.de\/foobar\/package/',
            (string) $history[1]['request']->getUri()
        );
        $this->assertContains('page=1', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('page=2', $history[1]['request']->getUri()->getQuery());
        $this->assertContains('json=1', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('json=1', $history[1]['request']->getUri()->getQuery());
        $this->assertContains('reset=1', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('reset=1', $history[1]['request']->getUri()->getQuery());
    }


    public function testCreatesAdaptersForPackages(): void
    {
        $this->mockNextResponses(
            [
                new Response(200, [], file_get_contents(__DIR__ . '/../../../Fixtures/TportalData/packageData.json')),
                new Response(200, [], json_encode([ 'PackageAccommodationInfos' => [] ]))
            ]
        );

        $adapters = [];
        $this->subject->fetchPackages(function($adapter) use (&$adapters) {
            $adapters[] = $adapter;
            $this->assertInstanceOf(ServiceAdapterInterface::class, $adapter);
        });

        $this->assertGreaterThan(0, \count($adapters), 'The callback should have been called more than once');
    }

    public function testAddsLanguageParameterToPackageRequestIfAvailable(): void
    {
        $history = [];
        $this->mockNextResponses(
            [
                new Response(200, [], file_get_contents(__DIR__ . '/../../../Fixtures/TportalData/packageData.json')),
                new Response(200, [], json_encode([ 'PackageAccommodationInfos' => [] ]))
            ],
            $history
        );

        $this->subject->setLanguage('fr');
        $this->subject->fetchPackages(function() {});

        $this->assertCount(2, $history, 'Api Client should have made 2 requests');
        $this->assertContains('lang=fr', $history[0]['request']->getUri()->getQuery());
        $this->assertContains('lang=fr', $history[1]['request']->getUri()->getQuery());
    }


    /**
     * @param $responses Response[]
     * @param array $history
     */
    private function mockNextResponses($responses, &$history = [], int $repeatLastResponse = 15): void
    {
        if ($repeatLastResponse > 0) {
            $lastResponse = array_pop($responses);
            for ($i = 0; $i < $repeatLastResponse; $i++) {
                $responses[] = $lastResponse;
            }
        }
        $mock = new MockHandler($responses);
        $stack = HandlerStack::create($mock);
        $stack->push(Middleware::history($history));

        $this->subject->setHttpClientSettings([ 'handler' => $stack ]);
    }

}
