<?php
namespace Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\FileAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;

/**
 * Tour adapter.
 *
 * Describes a tour.
 */
class TourAdapter extends AbstractObjectAdapter implements ArticleAdapterInterface
{
    /**
     * @return string
     */
    public function getExternalId()
    {
        return 'outdooractive_' . (string) $this->object->tour['id'];
    }

    /**
     * @return int
     */
    public function getMainType()
    {
        return ArticleConstants::TYPE_TOUR;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return (string) $this->object->tour->title;
    }

    /**
     * @return string|null
     */
    public function getAbstract()
    {
        $value = (string) $this->object->tour->shortText;
        if (!empty($value)) {
            return $value;
        }
    }

    /**
     * @return string|null
     */
    public function getDescription()
    {
        $value = (string) $this->object->tour->longText;
        if (!empty($value)) {
            return $value;
        }
    }

    /**
     * @return string|null
     */
    public function getStreet()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getZip()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getCity()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getCountry()
    {
        return null;
    }

    /**
     * @return float
     */
    public function getLatitude()
    {
        return (float) $this->object->tour->startingPoint['lat'];
    }

    /**
     * @return float
     */
    public function getLongitude()
    {
        return (float) $this->object->tour->startingPoint['lon'];
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses()
    {
        // Tours don't have additional address records.
        return [];
    }

    /**
     * @return CategoryAdapterInterface[]
     */
    public function getCategories()
    {
        return [
            (new CategoryAdapter($this->object->tour->category)),
        ];
    }

    /**
     * @return MediumAdapterInterface[]
     */
    public function getMedia()
    {
        $media = [];
        if ($this->object->tour->images->image) {
            /*
             * Get the id of the primary image first as not every image
             * has a primary="true" attribute.
             */
            $primaryImageId = (int) $this->object->tour->primaryImage['id'];
            $sorting = 2;

            foreach ($this->object->tour->images->image as $image) {
                $medium = new MediumAdapter($image);
                if ((int) $image['id'] == $primaryImageId) {
                    $medium->setSorting(1);
                } else {
                    $medium->setSorting($sorting);
                    $sorting++;
                }
                $media[] = $medium;
            }
        }
        return $media;
    }

    /**
     * @return FileAdapterInterface[]
     */
    public function getFiles()
    {
        return [];
    }

    /**
     * @return bool
     */
    public function hasAttributes()
    {
        return true;
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        $items = [
            new AttributeAdapter($this->getExternalId(), 'geometry', serialize($this->extractGeometry())),
        ];

        if ($this->object->tour->properties->property) {
            foreach ($this->object->tour->properties->property as $property) {
                $items[] = new AttributeAdapter(
                    $this->getExternalId(),
                    'properties',
                    (string) $property['tag']
                );
            }
        }

        if ($this->object->tour->labels && $this->object->tour->labels->attributes()) {
            foreach ($this->object->tour->labels->attributes() as $attribute => $value) {
                $items[] = new AttributeAdapter($this->getExternalId(), 'labels', $attribute);
            }
        }

        if ($this->object->tour->crossCountryTechnique) {
            foreach ($this->object->tour->crossCountryTechnique->property as $technique) {
                $items[] = new AttributeAdapter(
                    $this->getExternalId(),
                    'crossCountryTechnique',
                    (string) $technique['tag']
                );
            }
        }

        foreach (['jan', 'feb', 'mar', 'apr', 'may', 'jun', 'jul', 'aug', 'sep', 'oct', 'nov', 'dec'] as $month) {
            if (filter_var((string) $this->object->tour->season[$month], FILTER_VALIDATE_BOOLEAN)) {
                $items[] = new AttributeAdapter($this->getExternalId(), 'bestSeason', $month);
            }
        }

        $simpleAttributes = [
            'ascentElevation' => (int) $this->object->tour->elevation['ascent'],
            'conditionRating' => (int) $this->object->tour->rating['condition'],
            'descentElevation' => (int) $this->object->tour->elevation['descent'],
            'difficultyRating' => (int) $this->object->tour->rating['difficulty'],
            'equipment' => (string) $this->object->tour->equipment,
            'experienceRating' => (int) $this->object->tour->rating['qualityOfExperience'],
            'landscapeRating' => (int) $this->object->tour->rating['landscape'],
            'maximumAltitude' => (int) $this->object->tour->elevation['maxAltitude'],
            'minimumAltitude' => (int) $this->object->tour->elevation['minAltitude'],
            'publicTransit' => (string) $this->object->tour->publicTransit,
            'privateTransit' => (string) $this->object->tour->gettingThere,
            'startingPoint' => (string) $this->object->tour->startingPointDescr,
            'techniqueRating' => (int) $this->object->tour->rating['technique'],
            'tourDuration' => (int) $this->object->tour->time['min'],
            'tourLength' => (int) $this->object->tour->length,
            'dataSource' => 'outdooractive',
            'dataSourceId' => (string) $this->object->tour['id'],
        ];

        foreach ($simpleAttributes as $name => $data) {
            $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
        }

        return $items;
    }

    private function extractGeometry()
    {
        $geometry = [];
        if ($this->object->tour->geometry) {
            $geoItems = explode(' ', $this->object->tour->geometry);
            foreach ($geoItems as $geoItem) {
                $geo = explode(',', $geoItem);
                $geometry[] = [ (float) $geo[1], (float) $geo[0] ];
            }
        }
        return $geometry;
    }

    /**
     * @return string|null
     */
    public function getFacebookUri()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getTwitterUri()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getInstagramUri()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getYoutubeUri()
    {
        return null;
    }

    /**
     * @return string
     */
    public function getSourceName()
    {
        return (string) $this->object->tour->meta->source['name'];
    }

    /**
     * @return string|null
     */
    public function getAuthorName()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getBookingUri()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getDetailUri()
    {
        return null;
    }

    /**
     * The complete opening times data.
     *
     * @return string|null
     */
    public function getOpeningTimes()
    {
        return null;
    }

    /**
     * Return the star classification for hotels.
     *
     * @return int
     */
    public function getStarClassification()
    {
        return 0;
    }

    /**
     * Return the average rating on a scale of 0-100
     *
     * @return int
     */
    public function getAverageRating()
    {
        return 0;
    }

    /**
     * Return the number of ratings.
     *
     * @return int
     */
    public function getNumberOfRatings()
    {
        return 0;
    }
}
