<?php
namespace Newland\Toubiz\Api\Service\Tportal;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\LodgingAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\OfferAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\PackageAdapter;
use Newland\Toubiz\Api\Utility\ArrayUtility;

/**
 * Service for legacy Toubiz TPortal.
 *
 * Concrete implementation to communicate with the TPortal
 * which is providing data for TOMAS-bound entities.
 */
class ApiService extends AbstractService
{
    /**
     * @var string Base URI of API endpoint.
     */
    const BASE_URI = 'https://tportal.toubiz.de';

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [
        'Lodging' => LodgingAdapter::class
    ];

    /**
     * Fetch lodgings.
     *
     * @param callable $block
     * @return void
     */
    public function fetchLodgings(callable $block)
    {
        $page = 1;
        while (true) {
            $data = $this->sendLodgingRequest($page);
            if (!$data) {
                return;
            }

            foreach ($data['housedata'] as $id => $item) {
                $lodging = new LodgingAdapter([
                    'housedata' => $data['housedata'][$id],
                    'searchresult' => $data['searchresult'][$id][0],
                    'searchresult_details' => $data['searchresult_details'][$id]
                ]);
                $block($lodging);
            }

            $page++;
        }
    }

    public function fetchOffers(callable $block)
    {
        $page = 1;

        while (true) {
            $data = $this->sendOfferRequest($page);
            if (!$data) {
                return;
            }

            // If there is only one result then `offerData` is not an array of
            // data objects but the data object itself.
            if (ArrayUtility::isAssociative($data['offerData'])) {
                $data['offerData'] = [ $data['offerData'] ];
            }

            foreach ($data['offerData'] as $index => $offer) {
                $id = $offer['serviceID'];
                $item = new OfferAdapter([
                    'offer' => $offer,
                    'serviceData' => $data['serviceData'][$id],
                    'availability' => $data['ServiceAvailabilities'][$id],
                    'geoResult' => $data['geoResultData'][$index]
                 ]);
                $block($item);
            }

            $page++;
        }
    }


    private function sendOfferRequest(int $page)
    {
        $parameters = [
            ':clientName' => $this->clientName,
            ':page' => $page
        ];

        $url = str_replace(
            array_keys($parameters),
            array_values($parameters),
            '/:clientName/offer?reset=1&json=1&page=:page'
        );

        $response = $this->httpClient->request('GET', $url);
        if ($response->getStatusCode() == 200) {
            $data = json_decode($response->getBody(), true);

            if (is_array($data)
                && array_key_exists('offerData', $data)
                && count($data['offerData']) > 0) {
                return $data;
            } else {
                return null;
            }
        }

        return null;
    }

    public function fetchPackages(callable $block)
    {
        $page = 1;

        while (true) {
            $data = $this->sendPackageRequest($page);
            if (!$data) {
                return;
            }

            foreach ($data['PackageAccommodationInfos'] as $package) {
                $id = $package['packageID'];
                $item = new PackageAdapter([
                    'package' => $package,
                    'serviceData' => $data['serviceData'][$id],
                ]);
                $block($item);
            }

            $page++;
        }
    }

    private function sendPackageRequest(int $page)
    {
        $url = $this->urlTemplate(
            '/:clientName/package?reset=1&json=1&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );

        $response = $this->httpClient->request('GET', $url);
        $data = $response->getStatusCode() === 200 ? json_decode($response->getBody(), true) : [];

        if (array_key_exists('PackageAccommodationInfos', $data) && \count($data['PackageAccommodationInfos']) > 0) {
            return $data;
        }

        return null;
    }


    /**
     * Builds an array of data for one lodging from given full array.
     *
     * @param string $id
     * @param array $data
     * @return array
     */
    protected function buildLodgingItemArray($id, $data)
    {
        $url = $this->urlTemplate(
            '/:clientName/package?reset=1&json=1&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );

        $response = $this->httpClient->request('GET', $url);
        $data = $response->getStatusCode() === 200 ? json_decode($response->getBody(), true) : [];

        if (array_key_exists('PackageAccommodationInfos', $data) && \count($data['PackageAccommodationInfos']) > 0) {
            return $data;
        }

        return null;
    }


    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $page
     * @return array|bool
     */
    protected function sendLodgingRequest($page)
    {
        $url = $this->urlTemplate(
            '/:clientName/ukv/search?reset=1&json=1&ukv_result_order=3&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );

        $response = $this->httpClient->request('GET', $url);
        if ($response->getStatusCode() == 200) {
            $data = json_decode($response->getBody(), true);

            /*
             * Checking on housedata if data is present but this could
             * also be any other field that is being returned in the
             * array, as data is spread across multiple response segments.
             */
            if ($data['housedata'] && count($data['housedata']) > 0) {
                return $data;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    private function urlTemplate(string $url, array $replacements): string
    {
        return str_replace(
            array_keys($replacements),
            array_map('urlencode', array_values($replacements)),
            $url
        );
    }
}
