<?php
namespace Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;

/**
 * Tour adapter.
 *
 * Describes a tour.
 */
class TourAdapter extends AbstractObjectAdapter implements ArticleAdapterInterface
{
    /**
     * @return string
     */
    public function getExternalId()
    {
        return 'outdooractive_' . (string)$this->object->tour['id'];
    }

    /**
     * @return int
     */
    public function getMainType()
    {
        return ArticleConstants::TYPE_TOUR;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return (string)$this->object->tour->title;
    }

    /**
     * @return string|null
     */
    public function getAbstract()
    {
        $value = (string)$this->object->tour->shortText;
        if (!empty($value)) {
            return $value;
        }
    }

    /**
     * @return string|null
     */
    public function getDescription()
    {
        $value = (string)$this->object->tour->longText;
        if (!empty($value)) {
            return $value;
        }
    }

    /**
     * @return string|null
     */
    public function getStreet()
    {
        // Tours don't have a fixed address.
        return;
    }

    /**
     * @return string|null
     */
    public function getZip()
    {
        // Tours don't have a fixed address.
        return;
    }

    /**
     * @return string|null
     */
    public function getCity()
    {
        // Tours don't have a fixed address.
        return;
    }

    /**
     * @return string|null
     */
    public function getCountry()
    {
        // Tours don't have a fixed address.
        return;
    }

    /**
     * @return float
     */
    public function getLatitude()
    {
        return (float)$this->object->tour->startingPoint['lat'];
    }

    /**
     * @return float
     */
    public function getLongitude()
    {
        return (float)$this->object->tour->startingPoint['lon'];
    }

    /**
     * @return array<AddressAdapterInterface>
     */
    public function getAddresses()
    {
        // Tours don't have additional address records.
        return [];
    }

    /**
     * @return array<CategoryAdapterInterface>
     */
    public function getCategories()
    {
        return [
            (new CategoryAdapter($this->object->tour->category))
        ];
    }

    /**
     * @return array<MediumAdapter>
     */
    public function getMedia()
    {
        $media = [];
        if ($this->object->tour->images->image) {
            foreach ($this->object->tour->images->image as $image) {
                $media[] = (new MediumAdapter($image));
            }
        }
        return $media;
    }

    /**
     * @return array<FileAdapterInterface>
     */
    public function getFiles()
    {
        return [];
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        $properties = [];
        if ($this->object->tour->properties->property) {
            foreach ($this->object->tour->properties->property as $property) {
                $properties[] = (string)$property['tag'];
            }
        }

        $geometry = [];
        if ($this->object->tour->geometry) {
            $items = explode(' ', $this->object->tour->geometry);
            foreach ($items as $item) {
                $geo = explode(',', $item);
                $geometry[] = [(float)$geo[1], (float)$geo[0]];
            }
        }

        return [
            'ascentElevation'   => (int)$this->object->tour->elevation['ascent'],
            'bestSeason'        => [
                'jan' => filter_var((string)$this->object->tour->season['jan'], FILTER_VALIDATE_BOOLEAN),
                'feb' => filter_var((string)$this->object->tour->season['feb'], FILTER_VALIDATE_BOOLEAN),
                'mar' => filter_var((string)$this->object->tour->season['mar'], FILTER_VALIDATE_BOOLEAN),
                'apr' => filter_var((string)$this->object->tour->season['apr'], FILTER_VALIDATE_BOOLEAN),
                'may' => filter_var((string)$this->object->tour->season['may'], FILTER_VALIDATE_BOOLEAN),
                'jun' => filter_var((string)$this->object->tour->season['jun'], FILTER_VALIDATE_BOOLEAN),
                'jul' => filter_var((string)$this->object->tour->season['jul'], FILTER_VALIDATE_BOOLEAN),
                'aug' => filter_var((string)$this->object->tour->season['aug'], FILTER_VALIDATE_BOOLEAN),
                'sep' => filter_var((string)$this->object->tour->season['sep'], FILTER_VALIDATE_BOOLEAN),
                'oct' => filter_var((string)$this->object->tour->season['oct'], FILTER_VALIDATE_BOOLEAN),
                'nov' => filter_var((string)$this->object->tour->season['nov'], FILTER_VALIDATE_BOOLEAN),
                'dec' => filter_var((string)$this->object->tour->season['dec'], FILTER_VALIDATE_BOOLEAN)
            ],
            'conditionRating'   => (int)$this->object->tour->rating['condition'],
            'descentElevation'  => (int)$this->object->tour->elevation['descent'],
            'difficultyRating'  => (int)$this->object->tour->rating['difficulty'],
            'equipment'         => (string)$this->object->tour->equipment,
            'experienceRating'  => (int)$this->object->tour->rating['qualityOfExperience'],
            'geometry'          => $geometry,
            'landscapeRating'   => (int)$this->object->tour->rating['landscape'],
            'maximumAltitude'   => (int)$this->object->tour->elevation['maxAltitude'],
            'minimumAltitude'   => (int)$this->object->tour->elevation['minAltitude'],
            'properties'        => $properties,
            'publicTransit'     => (string)$this->object->tour->publicTransit,
            'privateTransit'    => (string)$this->object->tour->gettingThere,
            'startingPoint'     => (string)$this->object->tour->startingPointDescr,
            'techniqueRating'   => (int)$this->object->tour->rating['technique'],
            'tourDuration'      => (int)$this->object->tour->time['min'],
            'tourLength'        => (int)$this->object->tour->length,
            'dataSource'        => 'outdooractive',
            'dataSourceId'      => (string)$this->object->tour['id']
        ];
    }

    /**
     * @return string|null
     */
    public function getFacebookUri()
    {
        // Tours don't have social media links.
        return;
    }

    /**
     * @return string|null
     */
    public function getTwitterUri()
    {
        // Tours don't have social media links.
        return;
    }

    /**
     * @return string|null
     */
    public function getInstagramUri()
    {
        // Tours don't have social media links.
        return;
    }

    /**
     * @return string|null
     */
    public function getYoutubeUri()
    {
        // Tours don't have social media links.
        return;
    }

    /**
     * @return string
     */
    public function getSourceName()
    {
        return (string)$this->object->tour->meta->source['name'];
    }

    /**
     * @return string|null
     */
    public function getAuthorName()
    {
        // Tours don't have an author.
        return;
    }

    /**
     * @return string|null
     */
    public function getBookingUri()
    {
        // Tours can't be booked.
        return;
    }

    /**
     * @return string|null
     */
    public function getDetailUri()
    {
        // Tours don't have an external page URI.
        return;
    }
}
