<?php
namespace Newland\Toubiz\Api\Service\Sisag\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\TransportationFacilityAdapterInterface;

/**
 * Transportation facility object adapter.
 *
 * Describes e.g. a ski lift from a region.
 */
class TransportationFacilityAdapter extends AbstractObjectAdapter implements TransportationFacilityAdapterInterface
{
    /**
     * @var array Mapping of external types to interface types.
     */
    const TYPE_MAPPING = [
        '4er-Sessellift' => self::TYPE_4_SEATER_CHAIRLIFT,
        '6er-Sessellift' => self::TYPE_6_SEATER_CHAIRLIFT,
        'Gondelbahn'     => self::TYPE_GONDOLA,
        'Pendelbahn'     => self::TYPE_REVERSIBLE_AERIAL_TRAM,
        'Skilift'        => self::TYPE_SKI_LIFT,
        'Zahnradbahn'    => self::TYPE_COGWHEEL_RAILWAY
    ];

    /**
     * @var array Mapping of external statuses to interface statuses.
     */
    const STATUS_MAPPING = [
        1 => self::STATUS_CLOSED,
        2 => self::STATUS_OPEN,
        3 => self::STATUS_IN_PREPARATION,
        4 => self::STATUS_CLOSED_REASON_STORM,
        5 => self::STATUS_CLOSED_END_OF_SEASON,
        7 => self::STATUS_CLOSED_REASON_AVALANCHE_HAZARD,
        8 => self::STATUS_TEMPORARY_CLOSED
    ];

    /**
     * @return string
     */
    public function getExternalId()
    {
        return (string)$this->object->ID;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return (string)$this->object->NAME;
    }

    /**
     * @return string
     */
    public function getRegion()
    {
        return (string)$this->object->REGION['id'];
    }

    /**
     * @return int
     */
    public function getType()
    {
        $type = (string)$this->object->TYPNAME;
        return self::TYPE_MAPPING[$type];
    }

    /**
     * @return int
     */
    public function getStatus()
    {
        $status = (int)$this->object->STATUS;
        return self::STATUS_MAPPING[$status];
    }

    /**
     * @return int|null
     */
    public function getLength()
    {
        $length = (int)$this->object->LAENGE;
        if ($length) {
            return $length;
        } else {
            return null;
        }
    }

    /**
     * @return int|null
     */
    public function getTravelTime()
    {
        $minutes = (int)$this->object->FAHRZEIT;
        if ($minutes) {
            return $minutes * 60;
        } else {
            return null;
        }
    }

    /**
     * @return \DateTime
     */
    public function getUpdatedAt()
    {
        $timezone = (new \DateTimeZone('Europe/Zurich'));
        return \DateTime::createFromFormat(
            'd.m.Y H:i',
            (string)$this->object->AUSGABE,
            $timezone
        );
    }
}
