<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\PointOfInterestAttributes;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\ExternalIdAdapter;
use Newland\Toubiz\Api\ObjectAdapter\ExternalIdAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasAdditionalExternalIds;
use Newland\Toubiz\Api\ObjectAdapter\HasLanguageGroupingSeparateFromOriginalId;
use Newland\Toubiz\Api\ObjectAdapter\UriAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\ObjectAdapter\YoutubeVideoAdapter;
use Newland\Toubiz\Api\ObjectAdapter\VimeoVideoAdapter;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\LegacyToubizArticleAdapterCommon;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\UsesCategoryMap;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;
use Newland\Toubiz\Api\Utility\UrlUtility;
use Psr\Log\LoggerAwareTrait;
use function Safe\json_encode;

/**
 * Point of interest adapter.
 *
 * This represents an Article from the DB service.
 */
class PointOfInterestAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface,
    ArticleWithSocialMediaLinksAdapterInterface,
    HasLanguageGroupingSeparateFromOriginalId,
    HasAdditionalExternalIds
{
    use LanguageAware,
        LoggerAwareTrait,
        UsesFirstMediumAsMainMedium,
        LegacyToubizArticleAdapterCommon,
        UsesCategoryMap;


    /**
     * Group names are not translated via API, so we add generic translations.
     * @var array<string, string>
     */
    protected $featureListMapping = [
        'e4506d15ec7dd2200fb01c6e0484493e' => PointOfInterestAttributes::ACCEPTED_PAYMENTS,
        '4ad54417b84ec9ee6491b006f1448ff9' => PointOfInterestAttributes::FOR_KIDS,
        '6233065458a995d42a70ef99e6fdc434' => PointOfInterestAttributes::SPOKEN_LANGUAGES,
        '555eaacf11d1037df2d840df2ecee09b' => PointOfInterestAttributes::PRICE_SEGMENT,
    ];

    /** @var array<string, mixed> */
    protected $valueMappedFeatures = [
        'd9fb978199b5a88bc6205ed79e75c9f0' => [
            'attribute' => PointOfInterestAttributes::INFORMATION,
            'valueMap' => [
                // DE
                'Barrierefrei' => PointOfInterestAttributes::INFORMATION_HANDICAP_ACCESSIBLE,
                'PKW Parkplatz' => PointOfInterestAttributes::INFORMATION_PARKING,
                'Busparkplatz' => PointOfInterestAttributes::INFORMATION_BUS_PARKING,
                'Familienfreundlich' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_FAMILIES,
                'Seniorenfreundlich' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_SENIORS,
                'Schlechtwetter geeignet' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_BAD_WEATHER,
                'WLAN' => PointOfInterestAttributes::INFORMATION_WIFI,
                'Überregional bedeutend' => PointOfInterestAttributes::INFORMATION_BROAD_APPEAL,
                'Inhabergeführte Geschäfte' => PointOfInterestAttributes::INFORMATION_OWNER_MANAGED_SHOP,
                'Nette Toilette' => PointOfInterestAttributes::INFORMATION_NICE_WC,

                // EN
                'suitable for the disabled' => PointOfInterestAttributes::INFORMATION_HANDICAP_ACCESSIBLE,
                'suitable bad weather' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_BAD_WEATHER,
                'bus park' => PointOfInterestAttributes::INFORMATION_BUS_PARKING,
                'family-friendly' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_FAMILIES,
                'car park' => PointOfInterestAttributes::INFORMATION_PARKING,
                'Seniors friendly' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_SENIORS,

                // FR
                'adapté(e) aux handicapés' => PointOfInterestAttributes::INFORMATION_HANDICAP_ACCESSIBLE,
                'approprié intempéries' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_BAD_WEATHER,
                'favorable aux familles' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_FAMILIES,
                'parking' => PointOfInterestAttributes::INFORMATION_PARKING,
                'Conforme aux personnes âgées' => PointOfInterestAttributes::INFORMATION_SUITABLE_FOR_SENIORS,
            ]
        ]
    ];

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_ATTRACTION;
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        $media = $this->extractMediaFromGallery(
            function (array $data) {
                return new MediumAdapter($data);
            }
        );

        $video = null;

        $videoUrl = $this->getDataMapString('video') ?: '';
        if (strpos($videoUrl, 'youtu') !== false) {
            $video = YoutubeVideoAdapter::create($videoUrl);
        }

        if (strpos($videoUrl, 'vimeo') !== false) {
            $video = VimeoVideoAdapter::create($videoUrl);
        }

        if ($video !== null) {
            $media[] = $video;
        }

        return $media;
    }

    /**
     * @return AttributeAdapter[]
     */
    public function getAttributes(): array
    {
        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    protected function isUrlPlaceholder(string $value): bool
    {
        return ($value === 'http://' || $value === 'https://');
    }

    public function getOpeningTimes(): ?string
    {
        $keysToCheck = [
            'opentimesmain',
            'opentimes',
            'closetimes',
            'actiontimes',
        ];

        $openTimes = [];
        $dataExists = 0;
        $dataMap = $this->object['object']['data_map'];

        foreach ($keysToCheck as $key) {
            if (array_key_exists($key, $dataMap)) {
                $openTimeData = $dataMap[$key];
                if ($openTimeData['content']) {
                    $dataExists++;
                    $openTimes[$key] = $openTimeData['content'];
                }
            }
        }

        return $dataExists > 0 ? json_encode($openTimes) : null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        return $this->getOpeningTimes() === null ? null : ArticleConstants::OPENING_TIMES_FORMAT_LEGACY_ATTRACTION;
    }

    public function getLevelOfMaintenance(): int
    {
        return (int) $this->object['object']['data_map']['level']['content'];
    }

    protected function parseAttributes(): array
    {
        $attributes = (array) array_replace(
            $this->parseCommonToubizAttributes(),
            $this->getFeatureList(),
            $this->mapDataMapEntries($this->dataMapMapping),
            $this->getValueMappedFeatures()
        );

        $childrenList = $this->getChildrenList();
        foreach ($this->extensionAttributeMapping as $extension => $mapping) {
            foreach ($mapping as $normalized => $external) {
                $item = $childrenList[$extension]['object']['data_map'][$external] ?? null;
                if (!\is_array($item)) {
                    continue;
                }

                $value = $this->normalizeDataMapItem($item);
                if ($value === null) {
                    continue;
                }
                $attributes[$normalized] = $value;
            }
        }

        $discountCardsContent = $this->object['object']['data_map']['discount_card']['content'] ?? null;
        $discountCards = $discountCardsContent ? ArrayUtility::trimExplode(',', $discountCardsContent) : null;

        if ($discountCards) {
            $attributes[PointOfInterestAttributes::DISCOUNT_CARDS] = $discountCards;
        }

        return $attributes;
    }

    private function getChildrenList(): array
    {
        $children = [];
        foreach ($this->object['object_children_list'] ?? [] as $child) {
            foreach ($child as $name => $content) {
                $children[$name] = $content;
            }
        }
        return $children;
    }


    private function getValueMappedFeatures(): array
    {
        $features = [];

        foreach ($this->object['object']['data_map']['feature'] ?? [] as $feature) {
            $parent = $feature['parent_id'] ?? null;
            $attribute = $this->valueMappedFeatures[$parent]['attribute'] ?? null;
            $valueMap = $this->valueMappedFeatures[$parent]['valueMap'] ?? [ ];
            $rawValue = $feature['content'] ?? null;

            if ($attribute === null || $rawValue === null || empty($rawValue)) {
                continue;
            }

            $rawValue = trim(StringCleaner::asString((string) $rawValue));
            $mappedValue = $valueMap[$rawValue] ?? null;
            if ($mappedValue === null) {
                $this->logger->warning(sprintf(
                    'Cannot map %s/%s `%s` to a normalized value',
                    $attribute,
                    $parent,
                    $rawValue
                ));
                $mappedValue = $rawValue;
            }

            $features[$attribute] = $features[$attribute] ?? [];
            $features[$attribute][] = $mappedValue;
        }

        return $features;
    }


    private function getFeatureList(): array
    {
        $features = [];

        foreach ($this->object['object']['data_map']['feature'] ?? [] as $feature) {
            $parent = $feature['parent_id'] ?? null;
            $group = $this->featureListMapping[$parent] ?? null;
            $content = $feature['content'] ?? null;

            if ($group === null || $content === null) {
                continue;
            }

            if (!\array_key_exists($group, $features)) {
                $features[$group] = [];
            }

            if (!empty($feature)) {
                $features[$group][] = $content;
            }
        }

        return array_filter($features);
    }

    public function getFiles(): array
    {
        return [];
    }


    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    /**
     * @var array<string, string>
     */
    private $dataMapMapping = [
        PointOfInterestAttributes::FREE_ENTRANCE => 'prices_free',
        PointOfInterestAttributes::PRICE_INFORMATION => 'priceinfotext',
        PointOfInterestAttributes::PRICES => 'prices',
        PointOfInterestAttributes::TAG => 'tags',
        PointOfInterestAttributes::PANORAMA => 'panorama',
    ];

    /**
     * @var array<string, array<string, string>>
     */
    protected $extensionAttributeMapping = [
        'therme_extension' => [
            PointOfInterestAttributes::THERMAL_CLASSIFICATION => 'klassifizierung',
            PointOfInterestAttributes::THERMAL_POOLQUANTITYINSSIDE => 'anzahl_becken_innen',
            PointOfInterestAttributes::THERMAL_POOLAREAINSSIDE => 'flaeche_becken_innen',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATURESTARTINSIDE => 'temperatur_becken_innen_von',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATUREENDINSIDE => 'temperatur_becken_innen_bis',
            PointOfInterestAttributes::THERMAL_POOLQUANTITYOUTSIDE => 'anzahl_becken_aussen',
            PointOfInterestAttributes::THERMAL_POOLAREAOUTSIDE => 'flaeche_becken_aussen ',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATURESTARTOUTSIDE => 'temperatur_becken_aussen_von',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATUREENDOUTSIDE => 'temperatur_becken_aussen_bis',
            PointOfInterestAttributes::THERMAL_POOLFEATURES => 'ausstattung_badebereich',
            PointOfInterestAttributes::THERMAL_POOLFEATURESADDITIONAL => 'zusaetzliche_ausstattung_therme',
            PointOfInterestAttributes::THERMAL_WATERTYPE => 'wasser',
            PointOfInterestAttributes::THERMAL_WATERCLEANING => 'reinigung_wasser',
            PointOfInterestAttributes::THERMAL_WATERDEPTHAVERAGE => 'durchschnittliche_wassertiefe',
            PointOfInterestAttributes::THERMAL_HASNONSWIMMMERSEPERATION => 'abtrennung_nichtschwimmer',
            PointOfInterestAttributes::THERMAL_HASTHERAPYPOOL => 'therapiebecken',
            PointOfInterestAttributes::THERMAL_HASSPORTPOOL => 'sport_becken',
            PointOfInterestAttributes::THERMAL_HASUNDERWATERMUSIC => 'becken_unterwassermusik',
            PointOfInterestAttributes::THERMAL_HASNATUREBATHINGLAKE => 'naturbadesee',
            PointOfInterestAttributes::THERMAL_HASWHIRLPOOL => 'whirlpool',
            PointOfInterestAttributes::THERMAL_HASPOOLMASSAGEJET => 'becken_massageduesen',
            PointOfInterestAttributes::THERMAL_HASFLUXPOOL => 'stroemungsbecken',
            PointOfInterestAttributes::THERMAL_HASSEAPOOL => 'meeresbecken',
            PointOfInterestAttributes::THERMAL_HASSINGLECABIN => 'einzelkabinen',
            PointOfInterestAttributes::THERMAL_HASFAMILYCABIN => 'familienkabinen',
            PointOfInterestAttributes::THERMAL_HASWARDROBE => 'kleiderschraenke',
            PointOfInterestAttributes::THERMAL_HASLOCKER => 'schliessfaecher',
            PointOfInterestAttributes::THERMAL_HASHAIRDRYER => 'haartrockner',
            PointOfInterestAttributes::THERMAL_HASSHOWERS => 'duschen',
            PointOfInterestAttributes::THERMAL_HASEMERGENCYROOM => 'sanitaetsraum',
            PointOfInterestAttributes::THERMAL_HASTOWELRENTAL => 'handtuch_verleih',
            PointOfInterestAttributes::THERMAL_HASBATHROBERENTAL => 'bademantel_verleih',
            PointOfInterestAttributes::THERMAL_HASSHOP => 'shop',
            PointOfInterestAttributes::THERMAL_HASGASTRONOMY => 'gastronomie',
            PointOfInterestAttributes::THERMAL_SAUNA_FEATURES => 'saunaausstattung',
            PointOfInterestAttributes::THERMAL_SAUNA_ADDITIONALFEATURES => 'ausstattung_saunabereich',
            PointOfInterestAttributes::THERMAL_SAUNA_QUANTITY => 'anzahl_saunen',
            PointOfInterestAttributes::THERMAL_SAUNA_MINIMIUMTEMPERATURE => 'mindesttemperatur',
            PointOfInterestAttributes::THERMAL_SAUNA_MAXIMUMTEMPERATURE => 'maximaltemperatur',
            PointOfInterestAttributes::THERMAL_SAUNA_ISINFUSIONQUARTERHOURLY => 'viertelstuendlich',
            PointOfInterestAttributes::THERMAL_SAUNA_ISINFUSIONHALFHOURLY => 'halbstuendlich',
            PointOfInterestAttributes::THERMAL_SAUNA_ISINFUSIONHOURLY => 'stuendlich',
            PointOfInterestAttributes::THERMAL_SAUNA_INFUSIONINTERVAL => 'zeitliche_taktung',
            PointOfInterestAttributes::THERMAL_SAUNA_COLDAPLICATION => 'abkuehl_angebote',
            PointOfInterestAttributes::THERMAL_SAUNA_QUIETROOM => 'ruheraeume',
            PointOfInterestAttributes::THERMAL_SAUNA_QUIETROOMQUANTITY => 'anzahl_ruheraeme',
            PointOfInterestAttributes::THERMAL_SAUNA_SERVICE => 'service_sauna',
            PointOfInterestAttributes::THERMAL_WELLNESS_APPLICATIONS => 'wellness_anwendungen',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASMASSAGES => 'wellnessmassagen',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASEXOTIC => 'exotische_wellnessangebote',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASSOFTPACK => 'softpack',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASTUBBATH => 'wannenbaeder',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASPEELING => 'koerperpeeling',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASCOSMETIC => 'kosmetikangebote',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASRELAXATION => 'entspannungsangebote',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASPRIVATESPA => 'private_spa',
            PointOfInterestAttributes::THERMAL_FITNESS_INDOOR => 'fitness_indoor',
            PointOfInterestAttributes::THERMAL_FITNESS_OUTDOOR => 'fitness_outdoor',
            PointOfInterestAttributes::THERMAL_HASHOTEL => 'thermalHotel',
            PointOfInterestAttributes::THERMAL_CAMPERQUANTITY => 'anzahl_reisemobil',
            PointOfInterestAttributes::THERMAL_FAMILYCHILDAGEMINIMUM => 'mindestalter_kinder',
            PointOfInterestAttributes::THERMAL_ISFAMILYFRIENDLY => 'familiengeeignet',
            PointOfInterestAttributes::THERMAL_ISWHEELCHAIRACCESSIBLE => 'rollstuhlgerecht',
            PointOfInterestAttributes::THERMAL_ISACCESSIBLE => 'behindertengerecht',
        ],
        'barrierfree_new' => [
            PointOfInterestAttributes::ACCESSIBLE_IS90 => 'barrierfree_90',
            PointOfInterestAttributes::ACCESSIBLE_IS80 => 'barrierfree_80',
            PointOfInterestAttributes::ACCESSIBLE_IS70 => 'barrierfree_70',
            PointOfInterestAttributes::ACCESSIBLE_IS60 => 'barrierfree_60',
            PointOfInterestAttributes::ACCESSIBLE_ISFORDEAFS => 'barrierfree_ear',
            PointOfInterestAttributes::ACCESSIBLE_ISFORBLINDS => 'barrierfree_eye',
            PointOfInterestAttributes::ACCESSIBLE_ISFORDISORDERS => 'barrierfree_lern',
            PointOfInterestAttributes::ACCESSIBLE_TEXT => 'barrierfree_text',
            PointOfInterestAttributes::ACCESSIBLE_HASELEVATOR => 'barrierfree_lift',
            PointOfInterestAttributes::ACCESSIBLE_ELEVATORWIDTH => 'barrierfree_lift_width',
            PointOfInterestAttributes::ACCESSIBLE_HASWC => 'barrierfree_wc',
            PointOfInterestAttributes::ACCESSIBLE_WCWIDTH => 'barrierfree_wc_width',
            PointOfInterestAttributes::ACCESSIBLE_HASWCPARTWIDTH => 'barrierfree_wc_part_width',
            PointOfInterestAttributes::ACCESSIBLE_HASWCPART => 'barrierfree_wc_part',
        ],
    ];

    public function getCitySelectors(): array
    {
        $ids = $this->object['object']['data_map']['city_relation']['remote_ids'] ?? '';
        $selectors = [];

        foreach (explode('#', $ids) as $id) {
            if (!empty($id)) {
                $id = str_replace('location_', '', $id);
                $selectors[] = new ExternalIdSelector(ExternalIdType::TOUBIZ_LEGACY, strip_tags((string) $id));
            }
        }

        return $selectors;
    }

    /**
     * @return CategoryAdapterInterface[]
     */
    public function getCategories(): array
    {
        return $this->extractCategories(
            function (array $properties) {
                return new CategoryAdapter($properties);
            }
        );
    }

    /** @return UriAdapterInterface[] */
    public function getBookingUris(): array
    {
        $bookingLinksContent = $this->object['object']['data_map']['booking_links']['content'] ?? null;
        if (!$bookingLinksContent) {
            return [];
        }

        $document = new \DOMDocument();
        $document->loadHTML($bookingLinksContent);

        $bookingLinks = [];
        foreach ($document->getElementsByTagName('a') as $i => $link) {
            $href = trim((string) $link->getAttribute('href'), '/');
            $bookingLinks[] = new UriAdapter(
                sprintf('%s_%s_%d', $this->getExternalId(), $this->getLanguage(), $i),
                trim((string) $link->nodeValue) ?: null,
                UrlUtility::addProtocolIfMissing($href)
            );
        }

        return $bookingLinks;
    }
}
