<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Tests\Unit\Guzzle;

use GuzzleHttp\Promise\FulfilledPromise;
use GuzzleHttp\Promise\Promise;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\Promise\RejectionException;
use Newland\Toubiz\Api\Exception\InvalidReturnValueException;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use PHPUnit\Framework\TestCase;

class ConcurrentPaginatedRequestsTest extends TestCase
{

    public function testCallsCallbackWithPageNumber(): void
    {
        $fulfilled = [];
        $rejected = [];
        $pool = new ConcurrentPaginatedRequests(3, function(int $page) use (&$fulfilled, &$rejected) {
            if ($page < 9) {
                $fulfilled[] = $page;
                return new FulfilledPromise(null);
            }

            $rejected[] = $page;
            return new RejectedPromise(null);
        });

        $pool->start()->wait();

        $this->assertEquals([ 1, 2, 3, 4, 5, 6, 7, 8 ], $fulfilled);
    }

    public function testRespectsMaximumPageNumberPassedEvenIfPromisesKeepOnBeingFulfilled(): void
    {
        $calls = [];
        $pool = new ConcurrentPaginatedRequests(3, function(int $page) use (&$calls) {
            $calls[] = $page;
            return new FulfilledPromise(null);
        });

        $pool->setLastPage(6);
        $pool->start()->wait();

        $this->assertEquals([ 1, 2, 3, 4, 5, 6 ], $calls);
    }

    public function testItAbortsExecutionToPreventIfiniteLoops()
    {
        $lastPage = 0;
        (new ConcurrentPaginatedRequests(3, function(int $page) use (&$lastPage) {
            $lastPage = $page;
            return new FulfilledPromise(null);
        }))->start()->wait();

        $this->assertEquals($lastPage, 9999);
    }

    public function testThrowsExceptionIfHandlerDoesNotReturnPromise()
    {
        $this->expectException(InvalidReturnValueException::class);
        $this->expectExceptionMessageRegExp('/Promise/');
        (new ConcurrentPaginatedRequests(3, function() {
            return null;
        }))->start()->wait();
    }

    public function testResolvesPromiseWithArrayOfResults(): void
    {
        $pool = new ConcurrentPaginatedRequests(3, function(int $page) {
            if ($page < 9) {
                return new FulfilledPromise($page);
            }

            return new RejectedPromise(null);
        });

        $this->assertEquals(
            [ 1, 2, 3, 4, 5, 6, 7, 8 ],
            $pool->start()->wait(true)
        );
    }

    public function testRejectsPromiseIfExceptionThrown(): void
    {
        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('This is a test exception');
        (new ConcurrentPaginatedRequests(3, function() {
            throw new \Exception('This is a test exception');
        }))->start()->wait();
    }

    // See TBPOI-538
    public function testFetchesAllPagesEvenIfSomeTakeLongerToLoad(): void
    {
        // Request 10 pages, every third of which takes longer.
        $pool = new ConcurrentPaginatedRequests(3, function(int $page) use (&$fulfilled) {
            if ($page <= 9) {
                $fulfilled[] = $page;
                return $this->pretendAsyncPromise($page % 3 === 0 ? 2500000 : 0);
            }

            return new RejectedPromise(null);
        });

        $pool->start()->wait(false);
        $this->assertContains(1, $fulfilled);
        $this->assertContains(2, $fulfilled);
        $this->assertContains(3, $fulfilled);
        $this->assertContains(4, $fulfilled);
        $this->assertContains(5, $fulfilled);
        $this->assertContains(6, $fulfilled);
        $this->assertContains(7, $fulfilled);
        $this->assertContains(8, $fulfilled);
        $this->assertContains(9, $fulfilled);
    }

    /** @dataProvider providePages */
    public function testAlwaysRequestsPagesInOrder(int $numberOfPages, int $concurrency, int $maxWaitMicrosecs): void
    {
        // Request pages with random wait times inbetween.
        // Dataprovider gives a lot of similar data so the test runs a couple of times since it depends on randomnes.
        $pool = new ConcurrentPaginatedRequests($concurrency, function(int $page) use ($numberOfPages, $maxWaitMicrosecs, &$fulfilled) {
            if ($page <= $numberOfPages) {
                $fulfilled[] = $page;
                return $this->pretendAsyncPromise(random_int(0, $maxWaitMicrosecs));
            }

            throw new RejectionException('End reached');
        });

        $pool->promise()->wait(false);

        for ($i = 0; $i < $numberOfPages; $i++) {
            $this->assertEquals($i + 1, $fulfilled[$i]);
        }
    }

    public function providePages(): array
    {
        $data = [];
        for ($i = 0; $i < 50; $i++) {
            $data[] = [
                random_int(50, 150),
                random_int(1, 30),
                random_int(50000, 250000)
            ];
        }
        return $data;
    }

    /**
     * PHP is a inherently single-threaded language - guzzles promises only appear to be asynchronous &
     * multithreaded because it takes single chunks of work and schedules them on a event loop similar to
     * JS. The only concurrent logic that is permitted in PHP are curl requests.
     *
     * To simulate concurrent requests we create promises that are actually many small promises that each wait
     * for a little bit. This way we give guzzle the chance to schedule work inbetween similar to how requests
     * would be schedules.
     */
    private function pretendAsyncPromise(int $microsecs): PromiseInterface
    {
        $promise = new FulfilledPromise(null);
        $stepSize = 1000;
        $steps =  ceil($microsecs/$stepSize);
        for ($i = 0; $i < $steps; $i++) {
            $promise = $promise->then(function() use ($stepSize) {
                return new Promise(function() use ($stepSize) {
                    usleep($stepSize);
                    return null;
                });
            });
        }
        return $promise;
    }

}
