<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GastronomyAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\HasAdditionalExternalIds;
use Newland\Toubiz\Api\ObjectAdapter\HasKitchenTimes;
use Newland\Toubiz\Api\ObjectAdapter\HasLanguageGroupingSeparateFromOriginalId;
use Newland\Toubiz\Api\ObjectAdapter\YoutubeVideoAdapter;
use Newland\Toubiz\Api\ObjectAdapter\VimeoVideoAdapter;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\LegacyToubizArticleAdapterCommon;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;
use Newland\Toubiz\Api\Utility\UrlUtility;
use Psr\Log\LoggerAwareTrait;

/**
 * Gastronomy adapter.
 *
 * This represents an Article with mapping for the Tportal-specific gastronomy.
 */
class GastronomyAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface,
    ArticleWithSocialMediaLinksAdapterInterface,
    HasLanguageGroupingSeparateFromOriginalId,
    HasAdditionalExternalIds,
    HasKitchenTimes
{
    use LanguageAware,
        UsesFirstMediumAsMainMedium,
        LegacyToubizArticleAdapterCommon,
        LoggerAwareTrait;

    private const GASTRONOMY_MAPPING = [
        // EN
        'African' => GastronomyAttributes::GASTRONOMY_STYLE_AFRICAN,
        'American' => GastronomyAttributes::GASTRONOMY_STYLE_AMERICAN,
        'Arabian' => GastronomyAttributes::GASTRONOMY_STYLE_ARABIC,
        'Asian' => GastronomyAttributes::GASTRONOMY_STYLE_ASIAN,
        'French' => GastronomyAttributes::GASTRONOMY_STYLE_FRENCH,
        'Greek' => GastronomyAttributes::GASTRONOMY_STYLE_GREEK,
        'Haute cuisine' => GastronomyAttributes::GASTRONOMY_STYLE_HAUTE,
        'Homelike' => GastronomyAttributes::GASTRONOMY_STYLE_HOMELY,
        'Indian' => GastronomyAttributes::GASTRONOMY_STYLE_INDIAN,
        'International' => GastronomyAttributes::GASTRONOMY_STYLE_INTERNATIONAL,
        'Italian' => GastronomyAttributes::GASTRONOMY_STYLE_ITALIAN,
        'Japanese' => GastronomyAttributes::GASTRONOMY_STYLE_JAPANESE,
        'Mediterranean' => GastronomyAttributes::GASTRONOMY_STYLE_MEDITERAN,
        'Mexican' => GastronomyAttributes::GASTRONOMY_STYLE_MEXICAN,
        'Oriental' => GastronomyAttributes::GASTRONOMY_STYLE_ORIENTAL,
        'Regional' => GastronomyAttributes::GASTRONOMY_STYLE_REGIONAL,
        'Spanish' => GastronomyAttributes::GASTRONOMY_STYLE_SPANISH,
        'Swiss' => GastronomyAttributes::GASTRONOMY_STYLE_SWISS,
        'Traditional fare' => GastronomyAttributes::GASTRONOMY_STYLE_TRADITIONAL,
        'Turkish' => GastronomyAttributes::GASTRONOMY_STYLE_TURKISH,

        // DE
        'Afrikanisch' => GastronomyAttributes::GASTRONOMY_STYLE_AFRICAN,
        'Amerikanisch' => GastronomyAttributes::GASTRONOMY_STYLE_AMERICAN,
        'Arabisch' => GastronomyAttributes::GASTRONOMY_STYLE_ARABIC,
        'Asiatisch' => GastronomyAttributes::GASTRONOMY_STYLE_ASIAN,
        'Französisch' => GastronomyAttributes::GASTRONOMY_STYLE_FRENCH,
        'Griechisch' => GastronomyAttributes::GASTRONOMY_STYLE_GREEK,
        'Haute Cuisine' => GastronomyAttributes::GASTRONOMY_STYLE_HAUTE,
        'Gutbürgerlich' => GastronomyAttributes::GASTRONOMY_STYLE_HOMELY,
        'Indisch' => GastronomyAttributes::GASTRONOMY_STYLE_INDIAN,
        'Italienisch' => GastronomyAttributes::GASTRONOMY_STYLE_ITALIAN,
        'Japanisch' => GastronomyAttributes::GASTRONOMY_STYLE_JAPANESE,
        'Mediterran' => GastronomyAttributes::GASTRONOMY_STYLE_MEDITERAN,
        'Mexikanisch' => GastronomyAttributes::GASTRONOMY_STYLE_MEXICAN,
        'Orientalisch' => GastronomyAttributes::GASTRONOMY_STYLE_ORIENTAL,
        'Spanisch' => GastronomyAttributes::GASTRONOMY_STYLE_SPANISH,
        'Schweizerisch' => GastronomyAttributes::GASTRONOMY_STYLE_SWISS,
        'Traditionell' => GastronomyAttributes::GASTRONOMY_STYLE_TRADITIONAL,
        'Türkisch' => GastronomyAttributes::GASTRONOMY_STYLE_TURKISH,

        // FR
        'africain' => GastronomyAttributes::GASTRONOMY_STYLE_AFRICAN,
        'Américaines' => GastronomyAttributes::GASTRONOMY_STYLE_AMERICAN,
        'Arabe' => GastronomyAttributes::GASTRONOMY_STYLE_ARABIC,
        'Asiatique' => GastronomyAttributes::GASTRONOMY_STYLE_ASIAN,
        'Française' => GastronomyAttributes::GASTRONOMY_STYLE_FRENCH,
        'Grecque' => GastronomyAttributes::GASTRONOMY_STYLE_GREEK,
        'Bourgeoise' => GastronomyAttributes::GASTRONOMY_STYLE_HOMELY,
        'Indien' => GastronomyAttributes::GASTRONOMY_STYLE_INDIAN,
        'Internationale' => GastronomyAttributes::GASTRONOMY_STYLE_INTERNATIONAL,
        'Italienne' => GastronomyAttributes::GASTRONOMY_STYLE_ITALIAN,
        'Japonais' => GastronomyAttributes::GASTRONOMY_STYLE_JAPANESE,
        'Méditerranéenne' => GastronomyAttributes::GASTRONOMY_STYLE_MEDITERAN,
        'mexicaine' => GastronomyAttributes::GASTRONOMY_STYLE_MEXICAN,
        'Régionale' => GastronomyAttributes::GASTRONOMY_STYLE_REGIONAL,
        'Espagnole' => GastronomyAttributes::GASTRONOMY_STYLE_SPANISH,
        'suisses' => GastronomyAttributes::GASTRONOMY_STYLE_SWISS,
        'Cuisine traditionnelle' => GastronomyAttributes::GASTRONOMY_STYLE_TRADITIONAL,
        'Turc' => GastronomyAttributes::GASTRONOMY_STYLE_TURKISH,

        // ES
        'árabe' => GastronomyAttributes::GASTRONOMY_STYLE_ARABIC,
        'Asiática' => GastronomyAttributes::GASTRONOMY_STYLE_ASIAN,
        'Francesa' => GastronomyAttributes::GASTRONOMY_STYLE_FRENCH,
        'Griega' => GastronomyAttributes::GASTRONOMY_STYLE_GREEK,
        'Alta cocina' => GastronomyAttributes::GASTRONOMY_STYLE_HAUTE,
        'Casera' => GastronomyAttributes::GASTRONOMY_STYLE_HOMELY,
        'Hindú' => GastronomyAttributes::GASTRONOMY_STYLE_INDIAN,
        'Japonés' => GastronomyAttributes::GASTRONOMY_STYLE_JAPANESE,
        'Mediterránea' => GastronomyAttributes::GASTRONOMY_STYLE_MEDITERAN,
        'mexicana' => GastronomyAttributes::GASTRONOMY_STYLE_MEXICAN,
        'Española' => GastronomyAttributes::GASTRONOMY_STYLE_SPANISH,
        'suizas' => GastronomyAttributes::GASTRONOMY_STYLE_SWISS,
        'Cocina casera' => GastronomyAttributes::GASTRONOMY_STYLE_TRADITIONAL,

        // IT
        'Africano' => GastronomyAttributes::GASTRONOMY_STYLE_AFRICAN,
        'Americana' => GastronomyAttributes::GASTRONOMY_STYLE_AMERICAN,
        'Arabo' => GastronomyAttributes::GASTRONOMY_STYLE_ARABIC,
        'Asiatica' => GastronomyAttributes::GASTRONOMY_STYLE_ASIAN,
        'Francese' => GastronomyAttributes::GASTRONOMY_STYLE_FRENCH,
        'Greca' => GastronomyAttributes::GASTRONOMY_STYLE_GREEK,
        'Casalinga' => GastronomyAttributes::GASTRONOMY_STYLE_HOMELY,
        'Indiano' => GastronomyAttributes::GASTRONOMY_STYLE_INDIAN,
        'Italiana' => GastronomyAttributes::GASTRONOMY_STYLE_ITALIAN,
        'Giapponese' => GastronomyAttributes::GASTRONOMY_STYLE_JAPANESE,
        'Mediterranea' => GastronomyAttributes::GASTRONOMY_STYLE_MEDITERAN,
        'messicana' => GastronomyAttributes::GASTRONOMY_STYLE_MEXICAN,
        'Orientale' => GastronomyAttributes::GASTRONOMY_STYLE_ORIENTAL,
        'Regionale' => GastronomyAttributes::GASTRONOMY_STYLE_REGIONAL,
        'Spagnola' => GastronomyAttributes::GASTRONOMY_STYLE_SPANISH,
        'svizzere' => GastronomyAttributes::GASTRONOMY_STYLE_SWISS,
        'Casereccio' => GastronomyAttributes::GASTRONOMY_STYLE_TRADITIONAL,
        'Turco' => GastronomyAttributes::GASTRONOMY_STYLE_TURKISH,
    ];

    private const HABITS_ALLERGIES_MAPPING = [
        // EN
        'On demand' => GastronomyAttributes::HABITS_ALLERGIES_ON_DEMAND,
        'Organic/wholefood' => GastronomyAttributes::HABITS_ALLERGIES_BIO,
        'Gluten free food' => GastronomyAttributes::HABITS_ALLERGIES_GLUTEN_FREE,
        'Halal' => GastronomyAttributes::HABITS_ALLERGIES_HALAL,
        'Kosher' => GastronomyAttributes::HABITS_ALLERGIES_KOSHER,
        'Lactose free food' => GastronomyAttributes::HABITS_ALLERGIES_LACTOSE_FREE,
        'Low-calorie dishes' => GastronomyAttributes::HABITS_ALLERGIES_REDUCTION,
        'Spezial diets (stomach, liver, gall, diabetes)' => GastronomyAttributes::HABITS_ALLERGIES_SPECIAL_DIET,
        'Vegan food' => GastronomyAttributes::HABITS_ALLERGIES_VEGAN,
        'Vegetarian' => GastronomyAttributes::HABITS_ALLERGIES_VEGETARIAN,

        // DE
        'Auf Anfrage' => GastronomyAttributes::HABITS_ALLERGIES_ON_DEMAND,
        'Bio-, Vollwertküche' => GastronomyAttributes::HABITS_ALLERGIES_BIO,
        'Glutenfreie Kost' => GastronomyAttributes::HABITS_ALLERGIES_GLUTEN_FREE,
        'Koscher' => GastronomyAttributes::HABITS_ALLERGIES_KOSHER,
        'Lactosefreie Kost' => GastronomyAttributes::HABITS_ALLERGIES_LACTOSE_FREE,
        'Reduktionskost' => GastronomyAttributes::HABITS_ALLERGIES_REDUCTION,
        'Spezielle Diät (Magen, Leber, Galle, Diabetes)' => GastronomyAttributes::HABITS_ALLERGIES_SPECIAL_DIET,
        'Vegane Küche' => GastronomyAttributes::HABITS_ALLERGIES_VEGAN,
        'Vegetarische Küche' => GastronomyAttributes::HABITS_ALLERGIES_VEGETARIAN,

        // FR
        'Sur demande' => GastronomyAttributes::HABITS_ALLERGIES_ON_DEMAND,
        'Cuisine bio, aux céréales complètes' => GastronomyAttributes::HABITS_ALLERGIES_BIO,
        'Repas sans gluten' => GastronomyAttributes::HABITS_ALLERGIES_GLUTEN_FREE,
        'Cacher' => GastronomyAttributes::HABITS_ALLERGIES_KOSHER,
        'Repas sans lactose' => GastronomyAttributes::HABITS_ALLERGIES_LACTOSE_FREE,
        'Prix réduit' => GastronomyAttributes::HABITS_ALLERGIES_REDUCTION,
        'Regime spécial (gastrique, hépatique,bile, diabète)' =>
            GastronomyAttributes::HABITS_ALLERGIES_SPECIAL_DIET,
        'Cuisine végétalienne' => GastronomyAttributes::HABITS_ALLERGIES_VEGAN,
        'Végétarienne' => GastronomyAttributes::HABITS_ALLERGIES_VEGETARIAN,

        // ES
        'A petición' => GastronomyAttributes::HABITS_ALLERGIES_ON_DEMAND,
        'Biológica, integral' => GastronomyAttributes::HABITS_ALLERGIES_BIO,
        'Alimentos sin gluten' => GastronomyAttributes::HABITS_ALLERGIES_GLUTEN_FREE,
        'Alimentos sin lactosa' => GastronomyAttributes::HABITS_ALLERGIES_LACTOSE_FREE,
        'Dieta de reducción de peso' => GastronomyAttributes::HABITS_ALLERGIES_REDUCTION,
        'Dietas particulares (estómago, hígado, biliar, diabetes)' =>
            GastronomyAttributes::HABITS_ALLERGIES_SPECIAL_DIET,
        'Comida vegana' => GastronomyAttributes::HABITS_ALLERGIES_VEGAN,
        'Vegetariana' => GastronomyAttributes::HABITS_ALLERGIES_VEGETARIAN,

        // IT
        'Su richiesta' => GastronomyAttributes::HABITS_ALLERGIES_ON_DEMAND,
        'Biologica, integrale' => GastronomyAttributes::HABITS_ALLERGIES_BIO,
        'Cibi senza glutine' => GastronomyAttributes::HABITS_ALLERGIES_GLUTEN_FREE,
        'Cibi senza lattosio' => GastronomyAttributes::HABITS_ALLERGIES_LACTOSE_FREE,
        'Riduzioni' => GastronomyAttributes::HABITS_ALLERGIES_REDUCTION,
        'Diete specifiche (stomaco, fegato, cistifellea, diabete)' =>
            GastronomyAttributes::HABITS_ALLERGIES_SPECIAL_DIET,
        'Cucina vegana' => GastronomyAttributes::HABITS_ALLERGIES_VEGAN,
    ];

    private const KITCHEN_STYLE_MAPPING = [

        // EN
        'Breakfast/brunch' => GastronomyAttributes::KITCHEN_STYLE_BRUNCH,
        'Home-made cakes and gateaux' => GastronomyAttributes::KITCHEN_STYLE_CAKES,
        'Large selection of cocktails' => GastronomyAttributes::KITCHEN_STYLE_COCKTAIL,
        'Coffee & cake' => GastronomyAttributes::KITCHEN_STYLE_COFFEE_AND_CAKE,
        'Diet cuisine' => GastronomyAttributes::KITCHEN_STYLE_DIET,
        'Evening meal' => GastronomyAttributes::KITCHEN_STYLE_DINING,
        'Fish specialities' => GastronomyAttributes::KITCHEN_STYLE_FISH,
        'Fondue / Raclette / cheese' => GastronomyAttributes::KITCHEN_STYLE_FONDUE,
        'Game specialities' => GastronomyAttributes::KITCHEN_STYLE_GAME,
        'Ice-cream menu' => GastronomyAttributes::KITCHEN_STYLE_ICE,
        'Lunch menu' => GastronomyAttributes::KITCHEN_STYLE_LUNCH,
        'meat specialties' => GastronomyAttributes::KITCHEN_STYLE_MEAT,
        'Regional specialities' => GastronomyAttributes::KITCHEN_STYLE_REGIONAL,
        'Seasonal dishes' => GastronomyAttributes::KITCHEN_STYLE_SEASONAL,
        'Snack menu' => GastronomyAttributes::KITCHEN_STYLE_SNACK,
        'Steak menu' => GastronomyAttributes::KITCHEN_STYLE_STEAK,
        'Large selection of vegetarian dishes' => GastronomyAttributes::KITCHEN_STYLE_VEGETARIAN,
        'Large selection of wines' => GastronomyAttributes::KITCHEN_STYLE_WINE,

        // DE
        'Frühstück/Brunch' => GastronomyAttributes::KITCHEN_STYLE_BRUNCH,
        'Hausgemachte Kuchen und Torten' => GastronomyAttributes::KITCHEN_STYLE_CAKES,
        'Große Cocktailauswahl' => GastronomyAttributes::KITCHEN_STYLE_COCKTAIL,
        'Kaffee & Kuchen' => GastronomyAttributes::KITCHEN_STYLE_COFFEE_AND_CAKE,
        'Diätküche' => GastronomyAttributes::KITCHEN_STYLE_DIET,
        'Abendessen' => GastronomyAttributes::KITCHEN_STYLE_DINING,
        'Fischspezialitäten' => GastronomyAttributes::KITCHEN_STYLE_FISH,
        'Fondue / Raclette / Käse' => GastronomyAttributes::KITCHEN_STYLE_FONDUE,
        'Wildspezialitäten' => GastronomyAttributes::KITCHEN_STYLE_GAME,
        'Eiskarte' => GastronomyAttributes::KITCHEN_STYLE_ICE,
        'Mittagstisch' => GastronomyAttributes::KITCHEN_STYLE_LUNCH,
        'Fleischspezialitäten' => GastronomyAttributes::KITCHEN_STYLE_MEAT,
        'Regionale Spezialitäten' => GastronomyAttributes::KITCHEN_STYLE_REGIONAL,
        'Saisonale Gerichte' => GastronomyAttributes::KITCHEN_STYLE_SEASONAL,
        'Vesperkarte/Brotzeitkarte' => GastronomyAttributes::KITCHEN_STYLE_SNACK,
        'Steakkarte' => GastronomyAttributes::KITCHEN_STYLE_STEAK,
        'Große vegetarische Auswahl' => GastronomyAttributes::KITCHEN_STYLE_VEGETARIAN,
        'Große Weinauswahl' => GastronomyAttributes::KITCHEN_STYLE_WINE,

        // FR
        'Petit-déjeuner/brunch' => GastronomyAttributes::KITCHEN_STYLE_BRUNCH,
        'Gâteaux et tartes faits maison' => GastronomyAttributes::KITCHEN_STYLE_CAKES,
        'Grand choix de cocktails' => GastronomyAttributes::KITCHEN_STYLE_COCKTAIL,
        'Café et gâteaux' => GastronomyAttributes::KITCHEN_STYLE_COFFEE_AND_CAKE,
        'Cuisine diététique' => GastronomyAttributes::KITCHEN_STYLE_DIET,
        'Dîner' => GastronomyAttributes::KITCHEN_STYLE_DINING,
        'Spécialités de poisson' => GastronomyAttributes::KITCHEN_STYLE_FISH,
        'Fondue / Raclette / fromage' => GastronomyAttributes::KITCHEN_STYLE_FONDUE,
        'Spécialités de gibier' => GastronomyAttributes::KITCHEN_STYLE_GAME,
        'Glaces' => GastronomyAttributes::KITCHEN_STYLE_ICE,
        'Déjeuner' => GastronomyAttributes::KITCHEN_STYLE_LUNCH,
        'spécialités de viande' => GastronomyAttributes::KITCHEN_STYLE_MEAT,
        'Spécialités régionales' => GastronomyAttributes::KITCHEN_STYLE_REGIONAL,
        'Plats saisonniers' => GastronomyAttributes::KITCHEN_STYLE_SEASONAL,
        'Carte pains avec charcuterie/fromage' => GastronomyAttributes::KITCHEN_STYLE_SNACK,
        'Steaks' => GastronomyAttributes::KITCHEN_STYLE_STEAK,
        'Grand choix de plats végétariens' => GastronomyAttributes::KITCHEN_STYLE_VEGETARIAN,
        'Grand choix de vins' => GastronomyAttributes::KITCHEN_STYLE_WINE,

        // ES
        'Desayunos/brunchs' => GastronomyAttributes::KITCHEN_STYLE_BRUNCH,
        'Pastelería y repostería caseras' => GastronomyAttributes::KITCHEN_STYLE_CAKES,
        'Gran selección de cócteles' => GastronomyAttributes::KITCHEN_STYLE_COCKTAIL,
        'Café y pasteles' => GastronomyAttributes::KITCHEN_STYLE_COFFEE_AND_CAKE,
        'Cocina dietética' => GastronomyAttributes::KITCHEN_STYLE_DIET,
        'Cena' => GastronomyAttributes::KITCHEN_STYLE_DINING,
        'Especialidades en pescado' => GastronomyAttributes::KITCHEN_STYLE_FISH,
        'Fondue / Raclette / queso' => GastronomyAttributes::KITCHEN_STYLE_FONDUE,
        'Especialidades de caza' => GastronomyAttributes::KITCHEN_STYLE_GAME,
        'Carta de helados' => GastronomyAttributes::KITCHEN_STYLE_ICE,
        'Almuerzos de mediodía' => GastronomyAttributes::KITCHEN_STYLE_LUNCH,
        'especialidades de carne' => GastronomyAttributes::KITCHEN_STYLE_MEAT,
        'Especialidades regionales' => GastronomyAttributes::KITCHEN_STYLE_REGIONAL,
        'Platos de temporada' => GastronomyAttributes::KITCHEN_STYLE_SEASONAL,
        'Carta de platos ligeros, estilo merendero' => GastronomyAttributes::KITCHEN_STYLE_SNACK,
        'Carta de carnes' => GastronomyAttributes::KITCHEN_STYLE_STEAK,
        'Gran selección vegetariana' => GastronomyAttributes::KITCHEN_STYLE_VEGETARIAN,
        'Gran selección de vinos' => GastronomyAttributes::KITCHEN_STYLE_WINE,

        // ES
        'Clazione/brunch' => GastronomyAttributes::KITCHEN_STYLE_BRUNCH,
        'Torte e dolci fatti in casa' => GastronomyAttributes::KITCHEN_STYLE_CAKES,
        'Ampia scelta di cocktail' => GastronomyAttributes::KITCHEN_STYLE_COCKTAIL,
        'Caffè e dolcetti' => GastronomyAttributes::KITCHEN_STYLE_COFFEE_AND_CAKE,
        'Cucina dietetica' => GastronomyAttributes::KITCHEN_STYLE_DIET,
        'Specialità pesce' => GastronomyAttributes::KITCHEN_STYLE_FISH,
        'Fondue / Raclette / formaggio' => GastronomyAttributes::KITCHEN_STYLE_FONDUE,
        'Specialità selvaggina' => GastronomyAttributes::KITCHEN_STYLE_GAME,
        'Specialità gelati' => GastronomyAttributes::KITCHEN_STYLE_ICE,
        'Aperto a mezzogiorno' => GastronomyAttributes::KITCHEN_STYLE_LUNCH,
        'specialità di carne' => GastronomyAttributes::KITCHEN_STYLE_MEAT,
        'Specialità regionali' => GastronomyAttributes::KITCHEN_STYLE_REGIONAL,
        'Piatti di stagione' => GastronomyAttributes::KITCHEN_STYLE_SEASONAL,
        'Menu serale' => GastronomyAttributes::KITCHEN_STYLE_SNACK,
        'Specialità carne' => GastronomyAttributes::KITCHEN_STYLE_STEAK,
        'Ampia scelta vegetariana' => GastronomyAttributes::KITCHEN_STYLE_VEGETARIAN,
        'Ampia scelta di vini' => GastronomyAttributes::KITCHEN_STYLE_WINE,
    ];

    private const PRICE_SEGMENT_MAPPING = [
        'high' => GastronomyAttributes::PRICE_SEGMENT_HIGH,
        'low' => GastronomyAttributes::PRICE_SEGMENT_LOW,
        'medium' => GastronomyAttributes::PRICE_SEGMENT_MEDIUM,
        'hoch' => GastronomyAttributes::PRICE_SEGMENT_HIGH,
        'niedrig' => GastronomyAttributes::PRICE_SEGMENT_LOW,
        'mittel' => GastronomyAttributes::PRICE_SEGMENT_MEDIUM,
        'élevé' => GastronomyAttributes::PRICE_SEGMENT_HIGH,
        'faible' => GastronomyAttributes::PRICE_SEGMENT_LOW,
        'moyen' => GastronomyAttributes::PRICE_SEGMENT_MEDIUM,
        'alto' => GastronomyAttributes::PRICE_SEGMENT_HIGH,
        'bajo' => GastronomyAttributes::PRICE_SEGMENT_LOW,
        'medio' => GastronomyAttributes::PRICE_SEGMENT_MEDIUM,
        'alta' => GastronomyAttributes::PRICE_SEGMENT_HIGH,
        'basso' => GastronomyAttributes::PRICE_SEGMENT_LOW,
    ];

    private const FAMILIES_MAPPING = [
        'welcome' => GastronomyAttributes::FAMILIES_WELCOME,
        'books' => GastronomyAttributes::FAMILIES_BOOKS,
        'highchair' => GastronomyAttributes::FAMILIES_HIGHCHAIR,
        'menu' => GastronomyAttributes::FAMILIES_MENU,
        'corner' => GastronomyAttributes::FAMILIES_CORNER,
        'painting' => GastronomyAttributes::FAMILIES_PAINTING,
        'playground' => GastronomyAttributes::FAMILIES_PLAYGROUND,
        'toys' => GastronomyAttributes::FAMILIES_TOYS,
        'changing' => GastronomyAttributes::FAMILIES_CHANGING,
    ];

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_GASTRONOMY;
    }

    public function getDescription(): ?string
    {
        $description = $this->object['description']['gastro']['text']
            ?? $this->getDataMapString('description');

        return $description ? $this->stringCleaner->purifyHtml((string) $description) : null;
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /** @return CategoryAdapterInterface[] */
    public function getCategories(): array
    {
        $externalIds = [];
        $categories = [];

        foreach (array_values($this->object['category'] ?? []) as $data) {
            $adapter = new CategoryAdapter($data);
            $categories[] = $adapter;
            $externalIds[] = $adapter->getExternalId();
        }

        $mainCategory = $this->getMainCategory();

        if ($mainCategory) {
            if (!\in_array($mainCategory->getExternalId(), $externalIds, true)
                && !empty($mainCategory->getName())) {
                $categories[] = $mainCategory;
            }

            usort(
                $categories,
                function (CategoryAdapter $a, CategoryAdapter $b) use ($mainCategory) {
                    return $a->getExternalId() === $mainCategory->getExternalId() ? -1 : 1;
                }
            );
        }

        return $categories;
    }

    private function getMainCategory(): ?CategoryAdapterInterface
    {
        $mainCategoryData = $this->object['main_category'] ?? null;
        if (!$mainCategoryData) {
            return null;
        }
        return new CategoryAdapter($mainCategoryData);
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->cache('media', function () {
            return array_merge($this->extractImages(), $this->extractVideos());
        });
    }

    public function extractVideos(): array
    {
        $videos = [];

        foreach ($this->object['video'] ?? [] as $videoCollection) {
            if (is_array($videoCollection)) {
                foreach ($videoCollection as $videoToAdd) {
                    $video = null;
                    if ($videoToAdd && UrlUtility::isValidUrl($videoToAdd)) {
                        if (strpos($videoToAdd, 'youtu') !== false) {
                            $video = YoutubeVideoAdapter::create($videoToAdd);
                        }
                        if (strpos($videoToAdd, 'vimeo') !== false) {
                            $video = VimeoVideoAdapter::create($videoToAdd);
                        }
                        if ($video !== null) {
                            $videos[] = $video;
                        }
                    }
                }
            }
        }

        return $videos;
    }

    /** @return MediumAdapter[] */
    private function extractImages(): array
    {
        if (empty($this->object['images'] ?? [])) {
            return [];
        }

        $items = [];

        if (ArrayUtility::isAssociative($this->object['images'])) {
            foreach ($this->object['images'] as $imageCollection) {
                if (is_array($imageCollection)) {
                    foreach ($imageCollection as $imageToAdd) {
                        $items[] = $imageToAdd;
                    }
                } else {
                    $items[] = $imageCollection;
                }
            }
        } else {
            foreach ($this->object['images'] as $imageCollection) {
                $items[] = $imageCollection;
            }
        }

        $media = [];
        foreach ($items as $image) {
            $adapter = new MediumAdapter($image);
            if (UrlUtility::isValidUrl($adapter->getSourceUri())) {
                $media[] = $adapter;
            }
        }

        return $media;
    }

    /**
     * @return AttributeAdapterInterface[]
     */
    public function getAttributes(): array
    {
        $attributes = AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function (string $name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );

        $decorationKeys = [
            'house_decoration' => GastronomyAttributes::HOUSE_DECORATION_GENERAL,
            'house_decoration_aral' => GastronomyAttributes::HOUSE_DECORATION_ARAL,
            'house_decoration_dehoga' => GastronomyAttributes::HOUSE_DECORATION_DEHOGA,
            'house_decoration_fine' => GastronomyAttributes::HOUSE_DECORATION_FINE,
            'house_decoration_michelin' => GastronomyAttributes::HOUSE_DECORATION_MICHELIN,
            'house_decoration_millau' => GastronomyAttributes::HOUSE_DECORATION_MILLAU,
            'house_decoration_varta' => GastronomyAttributes::HOUSE_DECORATION_VARTA,
        ];
        foreach ($decorationKeys as $key => $name) {
            foreach ($this->extractDecorationAttributes($key) as $decoration) {
                $attributes[] = new AttributeAdapter($this->getExternalId(), $name, $decoration);
            }
        }

        return array_merge(
            $attributes,
            $this->extractAttributeWithValueMapping(
                'gastro_style',
                GastronomyAttributes::GASTRONOMY_STYLE,
                static::GASTRONOMY_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'gastro_diet',
                GastronomyAttributes::HABITS_ALLERGIES,
                static::HABITS_ALLERGIES_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'gastro_kitchen',
                GastronomyAttributes::KITCHEN_STYLE,
                static::KITCHEN_STYLE_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'gastro_pricesegment',
                GastronomyAttributes::PRICE_SEGMENT,
                static::PRICE_SEGMENT_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'gastro_property',
                GenericArticleAttributes::SERVICES,
                static::$SERVICES_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'child',
                GastronomyAttributes::FAMILIES,
                static::FAMILIES_MAPPING,
                true
            )
        );
    }


    private function extractAttributeWithValueMapping(
        string $toubizValueName,
        string $attributeName,
        array $mapping,
        bool $valueOnly = false
    ): array {
        $attributes = [];

        $values = (array) $this->extractAttributes($toubizValueName, $valueOnly);
        foreach ($values as $rawValue) {
            $rawValue = trim(StringCleaner::asString((string) $rawValue));
            $value = $mapping[ $rawValue ] ?? null;

            if ($value === null) {
                $this->logger->warning(sprintf(
                    'Cannot map %s/%s `%s` to a normalized value',
                    $attributeName,
                    $toubizValueName,
                    $rawValue
                ));
                $value = $rawValue;
            }

            $attributes[] = new AttributeAdapter($this->getExternalId(), $attributeName, $value);
        }

        return $attributes;
    }


    /** @return FileAdapter[] */
    public function getFiles(): array
    {
        if (!\is_array($this->object['files'])) {
            return [];
        }

        $files = [];
        foreach ($this->object['files'] as $file) {
            $files[] = (new FileAdapter($file));
        }
        return $files;
    }

    public function getBookingUris(): array
    {
        $bookingUris = [];
        foreach ($this->object['property']['gastro_lunchgate'] ?? [] as $lunchgate) {
            foreach ($lunchgate['properties'] ?? [] as $properties) {
                $properties['parentExternalId'] = $this->getExternalId() . '_' . $this->getLanguage();
                if (strpos($properties['value'] ?? '', '@') === false) {
                    $bookingUris[] = new UriAdapter($properties);
                }
            }
        }
        return $bookingUris;
    }

    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] === '1');
    }

    public function getOpeningTimes(): ?string
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getOpeningTimesFormat(): ?string
    {
        return ArticleConstants::OPENING_TIMES_FORMAT_LEGACY;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getCitySelectors(): array
    {
        $id = $this->object['id_location'] ?? null;

        if (\is_string($id)) {
            return [new ExternalIdSelector(ExternalIdType::TOUBIZ_LEGACY, $id)];
        }

        return [];
    }

    private function extractAttributes(string $key, bool $valueOnly = false): ?array
    {
        $attributes = $this->object[ $key ]
            ?? $this->object['property'][ $key ]
            ?? [];

        $normalizedAttributes = [];
        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }

            if (array_key_exists('properties', $attribute)) {
                foreach ($attribute['properties'] as $property) {
                    $normalizedAttributes[] = $property;
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return null;
        }

        return array_map(
            function (array $item) use ($valueOnly) {
                if ($valueOnly) {
                    return $item['value'];
                }

                // Most of the time, text is only used. But if it also has a value given, it needs to be concatenated.
                if (!empty($item['value'])) {
                    return $item['text'] . ': ' . $item['value'];
                }

                return $item['text'];
            },
            $normalizedAttributes
        );
    }

    private function extractDecorationAttributes(string $key): array
    {
        $attributes = $this->object[ $key ]
            ?? $this->object['decoration'][ $key ]
            ?? [];

        $normalizedAttributes = [];

        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }

            if ($attribute['properties'] ?? []) {
                foreach ($attribute['properties'] as $property) {
                    $normalizedAttributes[] = $property;
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return [];
        }

        return array_filter(
            array_map(
                static function (array $item) {
                    if (!array_key_exists('text', $item)) {
                        return null;
                    }
                    if (preg_match('/^(.+)###(.*)/s', $item['text'], $matches)) {
                        return [
                            'label' => trim($matches[1]),
                            'description' => trim(strip_tags($matches[2])),
                            'value' => implode('_', array_filter(explode('###', $item['value']))),
                        ];
                    }
                    return null;
                },
                $normalizedAttributes
            )
        );
    }

    protected function parseAttributes(): array
    {
        $attributes = $this->parseCommonToubizAttributes();
        $translated = [
            'gastro_ambient' => GastronomyAttributes::AMBIANCE,
            'gastro_group' => GastronomyAttributes::GROUPS,
            'gastro_person' => GastronomyAttributes::CAPACITY,
            'gastro_payment' => GastronomyAttributes::ACCEPTED_PAYMENTS,
            'gastro_language' => GastronomyAttributes::SPOKEN_LANGUAGES,
        ];
        $untranslated = [
            'panorama' => GastronomyAttributes::PANORAMA,
        ];

        foreach ($translated as $key => $name) {
            $attribute = $this->extractAttributes($key, false);
            if (!empty($attribute)) {
                $attributes[ $name ] = $attribute;
            }
        }

        foreach ($untranslated as $key => $name) {
            $attribute = $this->extractAttributes($key, true);
            if (!empty($attribute)) {
                $attributes[ $name ] = $attribute;
            }
        }

        $currentInformationAttribute = $this->extractCurrentInformationAttribute();
        if ($currentInformationAttribute) {
            $attributes['currentInformation'] = $currentInformationAttribute;
        }

        $searchWords = $this->extractSearchWords();
        if ($searchWords) {
            $attributes[ GenericArticleAttributes::ADDITIONAL_SEARCH_STRING ] = $searchWords;
        }

        return $attributes;
    }

    private function extractCurrentInformationAttribute(): string
    {
        $currentInformation = $this->object['description']['gastro_event']['text'] ?? '';
        $currentInformation = $this->stringCleaner->purifyHtml($currentInformation);
        $currentInformation = $this->stringCleaner->cleanWhiteSpace($currentInformation);

        return trim($currentInformation);
    }

    private function extractSearchWords(): ?string
    {
        $searchWords = $this->object['searchwords'] ?? [];
        if (\count($searchWords) === 0) {
            return null;
        }

        $words = array_map(
            function ($item) {
                return $item['text'] ?? '';
            },
            $searchWords
        );

        return implode(' ', $words);
    }

    /**
     * @inheritDoc
     */
    public function getKitchenTimes(): array
    {
        $openingTimes = $this->object['opentimes'] ?? null;
        if (!$openingTimes) {
            return [];
        }

        $times = [];
        foreach ($openingTimes['kitchen'] ?? [] as $index => $entry) {
            $adapter = new KitchenTimeAdapter($entry, $index, [ $this->getExternalId(), $this->getLanguage() ]);
            if (!$adapter->isEmpty()) {
                $times[] = $adapter;
            }
        }

        return $times;
    }
}
