<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Client;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectionException;
use GuzzleHttp\Psr7\Response;
use Newland\Toubiz\Api\Exception\InvalidServiceResponseException;
use Newland\Toubiz\Api\Utility\RetryAbortingError;
use Newland\Toubiz\Api\Utility\RetryPool;
use Newland\Toubiz\Api\Utility\TimeDelta;
use Psr\Http\Message\UriInterface;
use Psr\Log\LoggerAwareTrait;
use function Safe\json_decode;

/**
 * Abstract service class.
 */
abstract class AbstractService
{
    use LoggerAwareTrait;

    /**
     * Abstract constant, must be set in concrete classes.
     *
     * @var string Base URI for requests.
     */
    const DEFAULT_BASE_URI = '';

    /**
     * @var string The API key matching the client name.
     */
    protected $apiKey = '';

    /**
     * @var \GuzzleHttp\Client Client for requests.
     */
    protected $httpClient;

    /**
     * @var string The client name to fetch data for.
     */
    protected $clientName = '';

    /**
     * @var array various service-specific parameters.
     */
    protected $parameters = [];

    /**
     * The given delta in which to sync.
     * Setting this to a confined time frame (e.g. 2 days) will cause the service to
     * sync only entities that have been changed in that time frame.
     *
     * @var TimeDelta
     */
    protected $delta;

    /**
     * @var string
     */
    protected $baseUri;

    /**
     * @api
     * @param string|null $baseUri
     */
    public function __construct(string $baseUri = null)
    {
        if (empty(static::DEFAULT_BASE_URI)) {
            throw new \InvalidArgumentException('DEFAULT_BASE_URI must be declared but is not!');
        }

        $this->baseUri = $baseUri ?? static::DEFAULT_BASE_URI;
        $this->httpClient = new Client([ 'base_uri' => $baseUri ?? static::DEFAULT_BASE_URI ]);
        $this->delta = TimeDelta::create('10 years');
    }

    public function setHttpClientSettings(array $config): void
    {
        $config = (array) array_replace_recursive($this->httpClient->getConfig(), $config);
        $this->httpClient = new Client($config);
    }

    /**
     * Factory method to fetch data from a service.
     *
     * @param string $thing Thing to fetch, used as method name for the effective service.
     * @param callable $block Method block that is being executed for one thing.
     * @return void
     *
     * @deprecated Use `fetch*` methods directly. Magic methods are being phased out.
     */
    public function fetch($thing, callable $block): void
    {
        $methodName = 'fetch' . ucwords($thing);
        $this->$methodName($block);
    }

    /**
     * Sets the client name.
     *
     * @api
     * @param string $clientName
     * @return void
     */
    public function setClientName($clientName): void
    {
        $this->clientName = $clientName;
    }

    /**
     * Sets the api key.
     *
     * @api
     * @param string $apiKey
     * @return void
     */
    public function setApiKey($apiKey): void
    {
        $this->apiKey = $apiKey;
    }

    public function setParameters(array $parameters): void
    {
        $this->parameters = $parameters;
    }

    public function setDelta(TimeDelta $delta): void
    {
        $this->delta = $delta;
    }

    protected function jsonRequest(
        UriInterface $url,
        int $retriesIfFail,
        \Closure $validateData = null
    ): PromiseInterface {
        $pool = new RetryPool($retriesIfFail, $this->parameters['sleepSecondsOnError'] ?? 3.0);
        $pool->setLogger($this->logger);
        $pool->setDebuggingInformation([
            'service' => str_replace('Newland\\Toubiz\\Api\\Service\\', '', static::class),
            'url' => (string) $url
        ]);

        $this->logger->debug(
            (string) $url,
            [ 'baseUri' => (string) $this->httpClient->getConfig('base_uri') ]
        );

        return $pool->retryOnPromiseRejection(function () use ($url, $validateData) {
            return $this->httpClient
                ->requestAsync('GET', (string) $url, [ 'headers' => [ 'Accept' => 'application/json' ] ])
                ->then(function (Response $response) use ($validateData, $url) {
                    return $this->jsonResponse($response, $url, $validateData);
                });
        });
    }


    protected function jsonResponse(Response $response, UriInterface $url, \Closure $validateData = null): array
    {
        $validateData = $validateData ?? function () {
            return true;
        };

        if ($response->getStatusCode() !== 200) {
            throw new RejectionException(sprintf(
                'Non 200 status code return for %s: %s [%d]',
                $url,
                $response->getReasonPhrase(),
                $response->getStatusCode()
            ));
        }

        $contentType = $response->getHeader('Content-Type')[0] ?? '';
        if (strpos($contentType, 'application/json') === false) {
            new InvalidServiceResponseException(sprintf(
                'Expected API to return JSON data, but `%s` returned `%s` -'
                . ' If this API is operating on a whitelist basis, you probably need to whitelist'
                . ' your IP `%s` on the server %s. If that IPv4 address is already whitelisted you may'
                . ' also need to whitelist correct outbound IPv6 address for the current server.',
                $url,
                $contentType,
                gethostbyname((string) gethostname()),
                $url->getHost() ?: ((string) $this->httpClient->getConfig('base_uri'))
            ));
        }

        $dataRaw = (string) $response->getBody();
        $dataRaw = StringCleaner::asString($dataRaw);
        if ($dataRaw === '') {
            throw new InvalidServiceResponseException('API did not return any payload (body is empty string)');
        }

        try {
            $data = json_decode($dataRaw, true);
        } catch (\Exception $e) {
            throw new InvalidServiceResponseException(
                sprintf('Error decoding JSON data: %s - the data probably is not valid JSON.', $e->getMessage()),
                $e->getCode(),
                $e
            );
        }

        if ($validateData($data)) {
            return $data;
        }

        throw new RetryAbortingError(new RejectionException('JSON Body is not valid'));
    }
}
