<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter;

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\ExternalIdAdapter;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\WithCacheProperty;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Ramsey\Uuid\Uuid;

trait LegacyToubizArticleAdapterCommon
{
    use WithCacheProperty;

    public function getSourceSystem(): string
    {
        return self::SOURCE_TOUBIZ_LEGACY;
    }

    protected function setupLegacyArticleAdapterCommon(): void
    {
        $this->stringCleaner = new StringCleaner();
    }

    public function getExternalId(): string
    {
        return strip_tags((string) $this->object['remote_id']);
    }

    public function getLanguageGrouping(): ?string
    {
        // Use toubiz new mapping if exists
        $toubizNewId = $this->object['object']['data_map']['infosystem_id']['content'] ?? null;
        if ($toubizNewId) {
            return $toubizNewId;
        }

        $id = trim(strip_tags((string) ($this->object['object']['id'] ?? '')));
        if ($id) {
            return md5(static::class . 'id' . $id);
        }

        return null;
    }

    public function getName(): string
    {
        return substr(
            strip_tags((string) $this->object['name']),
            0,
            240
        );
    }

    public function getAbstract(): ?string
    {
        return $this->stringCleaner->purifyHtml((string) $this->getDataMapString('intro'));
    }

    public function getDescription(): ?string
    {
        return $this->stringCleaner->purifyHtml((string) $this->getDataMapString('description'));
    }


    public function getLatitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[0];
        }

        $latitude = $this->object['location']['Latitude'] ?? null;
        return $latitude ? (float) $latitude : null;
    }

    public function getLongitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[1];
        }

        $longitude = $this->object['location']['Longitude'] ?? null;
        return $longitude ? (float) $longitude : null;
    }


    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    /**
     * @deprecated
     * @return bool
     */
    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    protected function parseCommonToubizAttributes(): array
    {
        $attributes = [];

        $searchWords = trim($this->object['object']['data_map']['searchwords']['content'] ?? '');
        if ($searchWords) {
            $attributes[GenericArticleAttributes::ADDITIONAL_SEARCH_STRING] = $searchWords;
        }

        return $attributes;
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialUri('facebook');
    }

    public function getTwitterUri(): ?string
    {
        return $this->getSocialUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialUri('flickr');
    }

    public function getSourceName(): ?string
    {
        $source = $this->object['headline_quellmandant'] ?? null;
        return $source ? strip_tags($source) : null;
    }

    public function getAuthorName(): ?string
    {
        $author = $this->object['object']['author'] ?? null;
        return $author ? strip_tags($author) : null;
    }

    protected function getDataMapString(string $dataMapItemName): ?string
    {
        $value = $this->object['object']['data_map'][$dataMapItemName]['content'] ?? null;
        return $value ? (string) $value : null;
    }

    protected function getSocialUri(string $name): ?string
    {
        $value = $this->getDataMapString($name);
        if ($value === null) {
            $socialMedia = $this->object['social_media'] ?? [];
            $value = $this->findInArray(
                $socialMedia,
                'property',
                $name
            )['value'] ?? null;
        }

        if ($value === null) {
            return null;
        }

        $value = trim($value);

        if ($value === '') {
            return null;
        }

        if (!$this->validateUrl($value)) {
            return null;
        }

        return $value;
    }

    protected function validateUrl(string $url): ?bool
    {
        return (bool) preg_match(
            '/^(?:https?:\/\/)?[\w\.\-]+(?:\.[\w\-\.]+)+[\w\!\#\$\%\&\(\)\*\+\,\-\.\/\:\;\=\?\@\[\]\~]+$/',
            $url
        );
    }


    /**
     * Finds the first matching key/value pair inside an array of arrays.
     *
     * @example
     *      $arr = [
     *          [
     *              "descriptionTypeID" => "",
     *              "descriptionTypeName" => "description"
     *              "description" => "..."
     *          ]
     *      ];
     *      $descriptionArray = $this->findInArray($arr, 'descriptionTypeName', 'description');
     *
     * @param array $array
     * @param mixed $searchKey
     * @param mixed $searchValue
     * @return array|null
     */
    protected function findInArray(array $array, $searchKey, $searchValue)
    {
        $result = array_filter(
            $array,
            function ($item) use ($searchKey, $searchValue) {
                return ((array) $item)[$searchKey] == $searchValue;
            }
        );

        if (count($result)) {
            return (array) array_shift($result);
        }

        return null;
    }

    /**
     * Extracts data map entries into a single associative array.
     * The given entries array must be a mapping of the normalized keys to the
     * ones in the data map.
     *
     * @param array $entries
     * @return array
     */
    protected function mapDataMapEntries(array $entries): array
    {
        $attributes = [];
        foreach ($entries as $normalized => $external) {
            $value = $this->getDataMapEntry($external);
            if ($value !== null) {
                $attributes[$normalized] = $value;
            }
        }
        return $attributes;
    }

    /**
     * @return array|bool|int|string|string[]|null
     */
    protected function getDataMapEntry(?string $name)
    {
        $entry = $this->object['object']['data_map'][$name] ?? null;
        if (!\is_array($entry)) {
            return null;
        }
        return $this->normalizeDataMapItem($entry);
    }

    /**
     * @return array|bool|int|string|string[]|null
     */
    protected function normalizeDataMapItem(array $item)
    {
        $type = $item['datatypestring'];
        $content = $item['content'];

        switch ($type) {
            case 'ezobjectrelationlist':
            case 'ezxmltext':
                return $content ? (string) $content : null;
            case 'ezstring':
                return $content ? strip_tags((string) $content) : null;
            case 'ezselection':
            case 'ezkeyword':
                if (!is_string($content) || $content === '') {
                    return null;
                }
                $items = ArrayUtility::trimExplode(',', $content);
                if (empty($items) || ($items[0] ?? null) === '---') {
                    return null;
                }
                return $items;
            case 'ezinteger':
                return $content ? (int) $content : null;
            case 'ezboolean':
                if ($content === true) {
                    return true;
                }
                if ($content === false) {
                    return false;
                }
                return null;
            case 'ezmatrix':
                if (!is_array($content) || empty($content)) {
                    return null;
                }
                return (array) $content;
            case 'ezimage':
                return $content['reference'] ?? $content['medium'] ?? null;
            case 'ezurl':
                if ($content === null || $content === 'http://' || $content === 'https://') {
                    return null;
                }
                return (string) $content;
        }

        return null;
    }

    protected function extractMediaFromGallery(callable $adapterInstanciator): array
    {
        return $this->cache('media', function () use ($adapterInstanciator) {
            $raw = $this->object['object']['data_map']['gallery']['content'] ?? null;
            if (!\is_array($raw)) {
                return [];
            }

            // Skip importing media records without a source uri.
            $raw = array_filter($raw, function (array $item) {
                return !empty($item['reference']);
            });

            $media = [];
            foreach ($raw as $key => $image) {
                $media[] = $adapterInstanciator(array_merge($image, [ '_sorting' => (int) $key ]));
            }
            return $media;
        });
    }


    public function getAdditionalExternalIds(): array
    {
        $externalIds = [
            new ExternalIdAdapter(ExternalIdType::TOUBIZ_LEGACY_REMOTE_ID, (string) $this->object['remote_id']),
        ];

        $id = $this->object['object']['id'] ?? null;
        if ($id) {
            $externalIds[] = new ExternalIdAdapter(ExternalIdType::TOUBIZ_LEGACY, (string) $id);
        }

        $dataMapMapping = [
            ExternalIdType::TOMAS => 'tomas_id',
            ExternalIdType::OUTDOORACTIVE => 'alpstein_id',
        ];

        foreach ($dataMapMapping as $type => $dataMapKey) {
            $id = $this->getDataMapString($dataMapKey);
            if ($id) {
                $externalIds[] = new ExternalIdAdapter($type, $id);
            }
        }

        // Sometimes toubiz_id contains toubiz legacy ids and sometimes toubiz new ids.
        // If it is a uuid, then it's toubiz new.
        $toubizId = $this->getDataMapString('toubiz_id');
        if ($toubizId) {
            $type = Uuid::isValid($toubizId) ? ExternalIdType::TOUBIZ : ExternalIdType::TOUBIZ_LEGACY;
            $externalIds[] = new ExternalIdAdapter($type, $toubizId);
        }

        return $externalIds;
    }
}
