<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GastronomyAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\HasAdditionalExternalIds;
use Newland\Toubiz\Api\ObjectAdapter\HasLanguageGroupingSeparateFromOriginalId;
use Newland\Toubiz\Api\ObjectAdapter\YoutubeVideoAdapter;
use Newland\Toubiz\Api\ObjectAdapter\VimeoVideoAdapter;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\LegacyToubizArticleAdapterCommon;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;
use Newland\Toubiz\Api\Utility\Regex;
use Newland\Toubiz\Api\Utility\UrlUtility;

/**
 * Gastronomy adapter.
 *
 * This represents an Article with mapping for the Tportal-specific gastronomy.
 */
class GastronomyAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface,
    ArticleWithSocialMediaLinksAdapterInterface,
    HasLanguageGroupingSeparateFromOriginalId,
    HasAdditionalExternalIds
{
    use LanguageAware,
        UsesFirstMediumAsMainMedium,
        LegacyToubizArticleAdapterCommon;

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_GASTRONOMY;
    }

    public function getDescription(): ?string
    {
        $description = $this->object['description']['gastro']['text']
            ?? $this->getDataMapString('description');

        return $description ? $this->stringCleaner->purifyHtml((string) $description) : null;
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /** @return CategoryAdapterInterface[] */
    public function getCategories(): array
    {
        $externalIds = [];
        $categories = [];

        foreach (array_values($this->object['category'] ?? []) as $data) {
            $adapter = new CategoryAdapter($data);
            $categories[] = $adapter;
            $externalIds[] = $adapter->getExternalId();
        }

        $mainCategory = $this->getMainCategory();

        if ($mainCategory) {
            if (!\in_array($mainCategory->getExternalId(), $externalIds, true)
                && !empty($mainCategory->getName())) {
                $categories[] = $mainCategory;
            }

            usort(
                $categories,
                function (CategoryAdapter $a, CategoryAdapter $b) use ($mainCategory) {
                    return $a->getExternalId() === $mainCategory->getExternalId() ? -1 : 1;
                }
            );
        }

        return $categories;
    }

    private function getMainCategory(): ?CategoryAdapterInterface
    {
        $mainCategoryData = $this->object['main_category'] ?? null;
        if (!$mainCategoryData) {
            return null;
        }
        return new CategoryAdapter($mainCategoryData);
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->cache('media', function () {
            return array_merge($this->extractImages(), $this->extractVideos());
        });
    }

    public function extractVideos(): array
    {
        $videos = [];

        foreach ($this->object['video'] ?? [] as $videoCollection) {
            if (is_array($videoCollection)) {
                foreach ($videoCollection as $videoToAdd) {
                    $video = null;
                    if ($videoToAdd && UrlUtility::isValidUrl($videoToAdd)) {
                        if (strpos($videoToAdd, 'youtu') !== false) {
                            $video = YoutubeVideoAdapter::create($videoToAdd);
                        }
                        if (strpos($videoToAdd, 'vimeo') !== false) {
                            $video = VimeoVideoAdapter::create($videoToAdd);
                        }
                        if ($video !== null) {
                            $videos[] = $video;
                        }
                    }
                }
            }
        }

        return $videos;
    }

    /** @return MediumAdapter[] */
    private function extractImages(): array
    {
        if (empty($this->object['images'] ?? [])) {
            return [];
        }

        $items = [];

        if (ArrayUtility::isAssociative($this->object['images'])) {
            foreach ($this->object['images'] as $imageCollection) {
                if (is_array($imageCollection)) {
                    foreach ($imageCollection as $imageToAdd) {
                        $items[] = $imageToAdd;
                    }
                } else {
                    $items[] = $imageCollection;
                }
            }
        } else {
            foreach ($this->object['images'] as $imageCollection) {
                $items[] = $imageCollection;
            }
        }

        $media = [];
        foreach ($items as $image) {
            $adapter = new MediumAdapter($image);
            if (UrlUtility::isValidUrl($adapter->getSourceUri())) {
                $media[] = $adapter;
            }
        }

        return $media;
    }

    /**
     * @return AttributeAdapterInterface[]
     */
    public function getAttributes(): array
    {
        $attributes = AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function (string $name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );

        $decorationKeys = [
            'house_decoration' => GastronomyAttributes::HOUSE_DECORATION_GENERAL,
            'house_decoration_aral' => GastronomyAttributes::HOUSE_DECORATION_ARAL,
            'house_decoration_dehoga' => GastronomyAttributes::HOUSE_DECORATION_DEHOGA,
            'house_decoration_fine' => GastronomyAttributes::HOUSE_DECORATION_FINE,
            'house_decoration_michelin' => GastronomyAttributes::HOUSE_DECORATION_MICHELIN,
            'house_decoration_millau' => GastronomyAttributes::HOUSE_DECORATION_MILLAU,
            'house_decoration_varta' => GastronomyAttributes::HOUSE_DECORATION_VARTA,
        ];
        foreach ($decorationKeys as $key => $name) {
            foreach ($this->extractDecorationAttributes($key) as $decoration) {
                $attributes[] = new AttributeAdapter($this->getExternalId(), $name, $decoration);
            }
        }

        return $attributes;
    }

    /** @return FileAdapter[] */
    public function getFiles(): array
    {
        if (!\is_array($this->object['files'])) {
            return [];
        }

        $files = [];
        foreach ($this->object['files'] as $file) {
            $files[] = (new FileAdapter($file));
        }
        return $files;
    }

    public function getBookingUris(): array
    {
        $bookingUris = [];
        foreach ($this->object['property']['gastro_lunchgate'] ?? [] as $lunchgate) {
            foreach ($lunchgate['properties'] ?? [] as $properties) {
                $properties['parentExternalId'] = $this->getExternalId() . '_' . $this->getLanguage();
                if (strpos($properties['value'] ?? '', '@') === false) {
                    $bookingUris[] = new UriAdapter($properties);
                }
            }
        }
        return $bookingUris;
    }

    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] === '1');
    }

    public function getOpeningTimes(): ?string
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getOpeningTimesFormat(): ?string
    {
        return ArticleConstants::OPENING_TIMES_FORMAT_LEGACY;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getCitySelectors(): array
    {
        $id = $this->object['id_location'] ?? null;

        if (\is_string($id)) {
            return [ new ExternalIdSelector(ExternalIdType::TOUBIZ_LEGACY, $id) ];
        }

        return [];
    }

    private function extractAttributes(string $key, bool $valueOnly = false): ?array
    {
        $attributes = $this->object[$key]
            ?? $this->object['property'][$key]
            ?? [];

        $normalizedAttributes = [];
        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }

            if (array_key_exists('properties', $attribute)) {
                foreach ($attribute['properties'] as $property) {
                    $normalizedAttributes[] = $property;
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return null;
        }

        return array_map(
            function (array $item) use ($valueOnly) {
                if ($valueOnly) {
                    return $item['value'];
                }

                // Most of the time, text is only used. But if it also has a value given, it needs to be concatenated.
                if (!empty($item['value'])) {
                    return $item['text'] . ': ' . $item['value'];
                }

                return $item['text'];
            },
            $normalizedAttributes
        );
    }

    private function extractDecorationAttributes(string $key): array
    {
        $attributes = $this->object[$key]
            ?? $this->object['decoration'][$key]
            ?? [];

        $normalizedAttributes = [];

        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }

            if ($attribute['properties'] ?? []) {
                foreach ($attribute['properties'] as $property) {
                    $normalizedAttributes[] = $property;
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return [];
        }

        return array_filter(
            array_map(
                static function (array $item) {
                    if (!array_key_exists('text', $item)) {
                        return null;
                    }
                    if (preg_match('/^(.+)###(.*)/s', $item['text'], $matches)) {
                        return [
                            'label' => trim($matches[1]),
                            'description' => trim(strip_tags($matches[2])),
                            'value' => implode('_', array_filter(explode('###', $item['value']))),
                        ];
                    }
                    return null;
                },
                $normalizedAttributes
            )
        );
    }

    protected function parseAttributes(): array
    {
        $attributes = $this->parseCommonToubizAttributes();
        $translated = [
            'gastro_ambient' => GastronomyAttributes::AMBIANCE,
            'gastro_group' => GastronomyAttributes::GROUPS,
            'gastro_kitchen' => GastronomyAttributes::KITCHEN_STYLE,
            'gastro_style' => GastronomyAttributes::GASTRONOMY_STYLE,
            'gastro_person' => GastronomyAttributes::CAPACITY,
            'gastro_payment' => GastronomyAttributes::ACCEPTED_PAYMENTS,
            'gastro_pricesegment' => GastronomyAttributes::PRICE_SEGMENT,
            'gastro_property' => GenericArticleAttributes::SERVICES,
            'gastro_language' => GastronomyAttributes::SPOKEN_LANGUAGES,
            'gastro_diet' => GastronomyAttributes::HABITS_ALLERGIES,
        ];
        $untranslated = [
            'child' => GastronomyAttributes::FAMILIES,
            'panorama' => GastronomyAttributes::PANORAMA,
        ];

        foreach ($translated as $key => $name) {
            $attribute = $this->extractAttributes($key, false);
            if (!empty($attribute)) {
                $attributes[$name] = $attribute;
            }
        }

        foreach ($untranslated as $key => $name) {
            $attribute = $this->extractAttributes($key, true);
            if (!empty($attribute)) {
                $attributes[$name] = $attribute;
            }
        }

        $currentInformationAttribute = $this->extractCurrentInformationAttribute();
        if ($currentInformationAttribute) {
            $attributes['currentInformation'] = $currentInformationAttribute;
        }

        $searchWords = $this->extractSearchWords();
        if ($searchWords) {
            $attributes[GenericArticleAttributes::ADDITIONAL_SEARCH_STRING] = $searchWords;
        }

        return $attributes;
    }

    private function extractCurrentInformationAttribute(): string
    {
        $currentInformation = $this->object['description']['gastro_event']['text'] ?? '';
        $currentInformation = $this->stringCleaner->purifyHtml($currentInformation);
        $currentInformation = $this->stringCleaner->cleanWhiteSpace($currentInformation);

        return trim($currentInformation);
    }

    private function extractSearchWords(): ?string
    {
        $searchWords = $this->object['searchwords'] ?? [];
        if (\count($searchWords) === 0) {
            return null;
        }

        $words = array_map(
            function ($item) {
                return $item['text'] ?? '';
            },
            $searchWords
        );

        return implode(' ', $words);
    }
}
