<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Guzzle;

use GuzzleHttp\Promise\EachPromise;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\PromisorInterface;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\Promise\RejectionException;
use Newland\Toubiz\Api\Exception\InvalidReturnValueException;

/**
 * Helper that helps execute paginated requests concurrently.
 *
 * The given handler is responsible for making requests itself and must return a promise.
 *
 * The given handler is called with the page number. If the handler returns a resolving promise
 * then it is assumed that this page was valid and a new one is being requested.
 *
 * If the handler returns a promise that is rejected then no more requests are being generated
 * assuming that the end of the data has been reached.
 *
 * Initially the given handler is called `$concurrency` times with corresponding page numbers
 * (starting at 1). Every time the promise returned by the handler resolves successfully a new
 * call for $page + $concurrency is being issued. If the promise is rejected then this thread
 * of execution is aborted.
 *
 * As a downside of this approach is that to many requests will be issued: Because `$concurrency`
 * requests are issued at once, requests done in a pool will overshoot the maximum by up to
 * `$concurrency` pages.
 *
 * @example
 * $pool = new ConcurrentPaginatedRequests(5, function(int $page) {
 *      $this->httpClient->requestAsync('GET', $this->uri($page))->then(function($response) {
 *          if ($response->getStatusCode !== 200) {
 *              throw new PromiseRejectionException('End of data reached');
 *          }
 *
 *          // ...
 *      });
 * });
 *
 * $pool->start()->promise()->wait();
 */
class ConcurrentPaginatedRequests implements PromisorInterface
{
    /** @var PromiseInterface */
    protected $promise;

    /** @var int */
    protected $concurrency;

    /** @var \Closure */
    protected $handler;

    /** @var int */
    protected $lastPage = 9999;

    /** @var int */
    protected $lastRequestedPage = 0;

    /** @var int[] */
    protected $queue = [];

    /** @var array */
    protected $results = [];

    public function __construct(int $concurrency, \Closure $handler)
    {
        $this->concurrency = $concurrency;
        $this->handler = $handler;
    }

    public function getLastPage(): int
    {
        return $this->lastPage;
    }

    public function setLastPage(int $lastPage): void
    {
        $this->lastPage = $lastPage;
    }

    public function promise()
    {
        if ($this->promise === null) {
            $this->start();
        }
        return $this->promise;
    }

    public function start(): PromiseInterface
    {
        $each = new EachPromise($this->requestIterator(), [ 'concurrency' => $this->concurrency ]);
        $this->promise = $each->promise()->then(function () {
            return array_values($this->results);
        });
        return $this->promise;
    }


    private function requestIterator(): \Generator
    {
        while ($this->lastRequestedPage < $this->lastPage) {
            yield $this->doRequestPage($this->lastRequestedPage + 1);
        }
    }

    private function doRequestPage(int $page): PromiseInterface
    {
        $handler = $this->handler;
        $result = $handler($page);
        if (!($result instanceof PromiseInterface)) {
            throw new InvalidReturnValueException('Handler must return Promise instance');
        }

        $this->lastRequestedPage = $page;
        return $result->then(function ($result) use ($page) {
            $this->results[$page] = $result;
        }, function ($e) use ($page) {
            if ($this->isLastPageInformation($e)) {
                $this->setLastPageIfSmaller($page);
            } else {
                $this->promise->reject($e);
            }
        });
    }

    /**
     * @param \Throwable|mixed $e
     */
    private function isLastPageInformation($e): bool
    {
        if ($e === null) {
            return true;
        }

        if ($e instanceof \Throwable) {
            do {
                if ($e instanceof RejectionException) {
                    return true;
                }
                // phpcs:ignore Generic.CodeAnalysis.AssignmentInCondition.FoundInWhileCondition
            } while ($e = $e->getPrevious());
        }

        return false;
    }

    private function setLastPageIfSmaller(int $page): void
    {
        if ($page < $this->lastPage) {
            $this->lastPage = $page;
        }
    }
}
