<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\ObjectAdapter;

use Newland\Toubiz\Api\ObjectAdapter\Concern\MediumConstants;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;

class YoutubeVideoAdapter implements MediumAdapterInterface, MediumWithStaticPreviewUri
{
    public static function create(?string $url): ?YoutubeVideoAdapter
    {
        if ($url === null) {
            return null;
        }
        $videoId = static::extractVideoId($url);
        return $videoId ? new YoutubeVideoAdapter($url, $videoId) : null;
    }

    /** @var string */
    private $videoId;

    /** @var string */
    private $videoUrl;

    public function __construct(string $url, string $videoId)
    {
        $this->videoUrl = $url;
        $this->videoId = $videoId;
    }

    public function getExternalId(): string
    {
        return 'youtube_' . $this->videoId;
    }

    public function getSourceUri(): string
    {
        return $this->videoUrl;
    }

    public function getPreviewUri(): ?string
    {
        return 'https://i.ytimg.com/vi/' . $this->videoId . '/maxresdefault.jpg';
    }

    public function getTitle(): ?string
    {
        return null;
    }

    public function getCopyright(): ?string
    {
        return null;
    }

    public function getType(): string
    {
        return MediumConstants::TYPE_VIDEO;
    }


    protected static function extractVideoId(string $url): ?string
    {
        $parsed = parse_url($url);

        if ($parsed === false) {
            return null;
        }

        // Regular youtube urls: https://youtube.com/watch?v={ID_GOES_HERE}
        if (array_key_exists('query', $parsed)
            && preg_match('/v=(.+)/', $parsed['query'], $queryMatches)
        ) {
            return $queryMatches[1];
        }

        // New youtu.be urls: https://youtu.be/{ID_GOES_HERE}
        if (array_key_exists('host', $parsed)
            && $parsed['host'] === 'youtu.be'
        ) {
            return str_replace('/', '', $parsed['path']);
        }

        // Embed urls: https://www.youtube.com/embed/{ID_GOES_HERE}
        if (array_key_exists('path', $parsed)
            && preg_match('/\/embed\/(\w+)/', $parsed['path'], $pathMatches)
        ) {
            return $pathMatches[1];
        }

        return null;
    }
}
