<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\ApiV1;

use GuzzleHttp\Promise\EachPromise;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter\ArticleAdapter;

class ArticleService extends AbstractService
{
    use LanguageAware;
    const DEFAULT_BASE_URI = 'https://mein.toubiz.de';

    public function fetchArticles(callable $block): ServiceResult
    {
        $ids = $this->listAllIds();
        $total = count($ids);

        $pool = new EachPromise($this->requestDetailPages($ids), [
            'concurrency' => $this->parameters['concurrency'] ?? 10,
            'fulfilled' => function ($data) use ($block, $total) {
                $adapter = new ArticleAdapter($data, $this->parameters['type']);
                $adapter->setLogger($this->logger);
                $block($adapter, $total);
            }
        ]);

        $pool->promise()->wait();

        $result = new ServiceResult();
        $result->setAll($ids);
        return $result;
    }

    private function listAllIds(): array
    {
        $query = array_replace_recursive($this->parameters['request'] ?? [], [
            'api_token' => $this->apiKey,
            'minimal' => true,
            'language' => $this->language,
        ]);

        if ($this->delta !== null && $this->delta->asYears() < 1) {
            $query['filter'] = $query['filter'] ?? [];
            $query['filter']['updatedAfter'] = (new \DateTime())->sub($this->delta)->format('Y-m-d\TH:i:s');
        }

        $uri = (new Uri('api/v1/article'))->withQuery(http_build_query($query));

        return $this->jsonRequest($uri, 5, function (array $data) {
            return empty($data['_errors']);
        })->then(function (array $data) {
            return array_keys($data['payload']);
        })->wait();
    }

    private function requestDetailPages(array $ids): \Generator
    {
        $query = [
            'language' => $this->language,
            'api_token' => $this->apiKey,
            'include' => [
                'emails',
                'contactInformation',
                'fieldValues',
                'fieldBlueprints',
                'phoneNumbers',
                'additionalAddress',
                'tags',
                'media',
                'seo',
                'openingTimes',
                'files',
                'permissions',
                'languages',
                'relatedArticles',
                'collections',
                'externalIds',
            ],
            'sizes' => [
                'preview' => $this->parameters['previewSize'] ?? '600'
            ]
        ];

        foreach ($ids as $id) {
            $uri = (new Uri(sprintf('api/v1/article/%s', $id)))->withQuery(http_build_query($query));
            yield $this->jsonRequest($uri, 5, function ($data) {
                return empty($data['_errors']);
            })->then(function ($data) {
                return $data['payload'];
            });
        }
    }
}
