<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DirectMarketerApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\DirectMarketerAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\HasAdditionalExternalIds;
use Newland\Toubiz\Api\ObjectAdapter\HasLanguageGroupingSeparateFromOriginalId;
use Newland\Toubiz\Api\ObjectAdapter\YoutubeVideoAdapter;
use Newland\Toubiz\Api\ObjectAdapter\VimeoVideoAdapter;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\AttributeAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\LegacyToubizArticleAdapterCommon;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;
use Newland\Toubiz\Api\Utility\UrlUtility;

/**
 * Direct marketer adapter.
 *
 * This represents an Article with mapping for the Toubiz specific direct marketer.
 */
class DirectMarketerAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface,
    ArticleWithSocialMediaLinksAdapterInterface,
    HasLanguageGroupingSeparateFromOriginalId,
    HasAdditionalExternalIds
{
    use LanguageAware;
    use UsesFirstMediumAsMainMedium;
    use LegacyToubizArticleAdapterCommon;

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_DIRECT_MARKETER;
    }

    public function getExternalId(): string
    {
        return strip_tags((string) $this->object['remote_id']);
    }

    public function getName(): string
    {
        return strip_tags((string) $this->object['name']);
    }

    public function getDescription(): ?string
    {
        $description = null;
        $value = '';
        if (!\is_array($this->object['description'])) {
            return null;
        }
        if (array_key_exists('direkt_event', $this->object['description'])) {
            $value .= $this->object['description']['direkt_event']['text'];
        }
        if (array_key_exists('direkt', $this->object['description'])) {
            $value .= $this->object['description']['direkt']['text'];
        }


        if (!empty($value)) {
            $description = $value;
        }

        return $description ? $this->stringCleaner->purifyHtml((string) $description) : null;
    }

    public function getLatitude(): ?float
    {
        $latitude = $this->object['location']['Latitude'] ?? null;
        return $latitude ? (float) $latitude : null;
    }

    public function getLongitude(): ?float
    {
        $longitude = $this->object['location']['Longitude'] ?? null;
        return $longitude ? (float) $longitude : null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [ $category ];
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->cache('media', function () {
            return array_merge($this->extractImages(), $this->extractVideos());
        });
    }

    public function extractVideos(): array
    {
        $videos = [];

        foreach ($this->object['video'] ?? [] as $videoCollection) {
            if (is_array($videoCollection)) {
                foreach ($videoCollection as $videoToAdd) {
                    $video = null;
                    if ($videoToAdd && UrlUtility::isValidUrl($videoToAdd)) {
                        if (strpos($videoToAdd, 'youtu') !== false) {
                            $video = YoutubeVideoAdapter::create($videoToAdd);
                        }
                        if (strpos($videoToAdd, 'vimeo') !== false) {
                            $video = VimeoVideoAdapter::create($videoToAdd);
                        }
                        if ($video !== null) {
                            $videos[] = $video;
                        }
                    }
                }
            }
        }

        return $videos;
    }

    /** @return MediumAdapter[] */
    private function extractImages(): array
    {
        if (empty($this->object['images'] ?? [])) {
            return [];
        }

        if (array_key_exists('all', $this->object['images'])) {
            // Some images are nested inside 'all'.
            $items = $this->object['images']['all'];
        } else {
            // ... and some are not inside 'all'.
            $items = $this->object['images'];
        }

        $media = [];
        foreach ($items as $image) {
            $adapter = new MediumAdapter($image);
            if (UrlUtility::isValidUrl($adapter->getSourceUri())) {
                $media[] = $adapter;
            }
        }

        return $media;
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    public function getAttributes(): array
    {
        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialMediaUri('facebook');
    }


    public function getTwitterUri(): ?string
    {
        return $this->getSocialMediaUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialMediaUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialMediaUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialMediaUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialMediaUri('flickr');
    }

    private function getSocialMediaUri(string $socialMedia): ?string
    {
        if (!array_key_exists('social_media', $this->object)
            || !\is_array($this->object['social_media'])
        ) {
            return null;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            $socialMedia
        );

        return $data ? $data['value'] : null;
    }

    public function getSourceName(): ?string
    {
        $source = $this->object['headline_quellmandant'] ?? null;
        return $source ? strip_tags((string) $source) : null;
    }

    public function getAuthorName(): ?string
    {
        $author = $this->object['author'] ?? null;
        return $author ? strip_tags((string) $author) : null;
    }

    public function getBookingUris(): array
    {
        return [];
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] ?? null) === '1';
    }

    public function getOpeningTimes(): ?string
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getOpeningTimesFormat(): ?string
    {
        return ArticleConstants::OPENING_TIMES_FORMAT_LEGACY;
    }

    private function parseAttributes(): array
    {
        $attributes = $this->parseCommonToubizAttributes();
        $keys = [
            'dv_marketing' => DirectMarketerAttributes::SHOPPING,
            'dv_prod_cat_aufstriche' => DirectMarketerAttributes::OIL,
            'dv_prod_cat_gemuese_obst' => DirectMarketerAttributes::VEGETABLES,
            'dv_prod_cat_getraenke' => DirectMarketerAttributes::DRINKS,
            'dv_prod_cat_saison' => DirectMarketerAttributes::SEASONALS,
            'dv_special' => GenericArticleAttributes::SERVICES,
            'dv_prod_cat_fleisch' => DirectMarketerAttributes::MEATS,
            'dv_prod_cat_gefluegel_fisch' => DirectMarketerAttributes::FISH,
            'dv_prod_cat_milch' => DirectMarketerAttributes::MILK,
            'dv_prod_cat_getreide' => DirectMarketerAttributes::CEREALS,
            'dv_prod_cat_handwerk' => DirectMarketerAttributes::CRAFTS,
            'dv_decoration' => DirectMarketerAttributes::AWARDS,
            'dv_decoration_regional' => DirectMarketerAttributes::REGIONAL_AWARDS,
        ];

        $valueOnlyKeys = [
            'panorama' => DirectMarketerAttributes::PANORAMA,
        ];

        foreach ($keys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key);
        }

        foreach ($valueOnlyKeys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key, true);
        }

        return $attributes;
    }

    private function extractAttributes(string $key, bool $valueOnly = false): ?array
    {
        $attributes = $this->object[$key]
            ?? $this->object['property'][$key]
            ?? $this->object['classification'][$key]
            ?? [];

        $normalizedAttributes = [];

        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }
            if (array_key_exists('properties', $attribute)) {
                foreach ($attribute['properties'] as $property) {
                    $normalizedAttributes[] = $property;
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return null;
        }

        return array_map(
            function (array $item) use ($valueOnly) {
                if ($valueOnly) {
                    return $item['value'];
                }
                return $item['text'];
            },
            $normalizedAttributes
        );
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getAbstract(): ?string
    {
        return null;
    }

    public function getFiles(): array
    {
        return [];
    }

    public function getCitySelectors(): array
    {
        $id = $this->object['id_location'] ?? null;

        if (\is_string($id)) {
            return [ new ExternalIdSelector(ExternalIdType::TOUBIZ_LEGACY, $id) ];
        }

        return [];
    }
}
