<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter;

use Newland\GpsFileParsing\Helper\XmlFileReader;
use Newland\GpsFileParsing\Model\Point;
use Newland\GpsFileParsing\ParserFactory;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GastronomyAttributes;
use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\MonthConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\PointOfInterestAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\ExternalIdAdapter;
use Newland\Toubiz\Api\ObjectAdapter\HasAdditionalExternalIds;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\VimeoVideoAdapter;
use Newland\Toubiz\Api\ObjectAdapter\YoutubeVideoAdapter;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Psr\Log\LoggerAwareTrait;

class ArticleAdapter extends AbstractObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface,
    HasAdditionalExternalIds,
    ArticleWithSocialMediaLinksAdapterInterface
{
    use LanguageAware, LoggerAwareTrait;

    /** @var int */
    protected $type;

    /** @var array|null */
    protected $cachedGeometry;

    public function __construct(array $adaptee, int $type)
    {
        $this->type = $type;
        parent::__construct($adaptee);
    }

    public function getSourceSystem(): string
    {
        return self::SOURCE_TOUBIZ;
    }

    public function getMainType(): int
    {
        return $this->type;
    }

    public function getName(): string
    {
        return $this->object['name'];
    }

    public function getAbstract(): ?string
    {
        return $this->object['abstract'];
    }

    public function getDescription(): ?string
    {
        return $this->object['description'];
    }

    public function getLatitude(): ?float
    {
        return $this->object['point']['latitude']
            ?? $this->object['area']['latitude']
            ?? $this->extractGeometry()[0][0]
            ?? null;
    }

    public function getLongitude(): ?float
    {
        return $this->object['point']['longitude']
            ?? $this->object['area']['longitude']
            ?? $this->extractGeometry()[0][1]
            ?? null;
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        $address = $this->object['point']['address']
            ?? $this->object['area']['address']
            ?? $this->object['relatedArticles']['tourContact'][0]['address']
            ?? null;

        if ($address) {
            $article = $this->object;
            $article['mainAddress'] = $address;
            return new MainAddressAdapter($article);
        }

        return null;
    }

    public function getAddresses(): array
    {
        $address = $this->getMainAddress();
        return $address ? [ $address ] : [];
    }

    public function getCategories(): array
    {
        // Indexing by id to prevent duplicate categories in primary & secondary
        $categories = [
            $this->object['primaryCategory']['id'] => $this->object['primaryCategory'],
        ];

        foreach ($this->object['secondaryCategories'] ?? [] as $item) {
            $categories[$item['id']] = $item;
        }

        return array_map(function ($item) {
            return new CategoryAdapter($item);
        }, array_values($categories));
    }

    public function getMedia(): array
    {
        $media = array_map(
            function ($item) {
                return new MediumAdapter($item);
            },
            $this->object['media'] ?? []
        );

        $youtubeVideo = YoutubeVideoAdapter::create($this->object['contactInformation']['youtubeVideo'] ?? null);
        if ($youtubeVideo) {
            $media[] = $youtubeVideo;
        }

        $vimeoVideo = VimeoVideoAdapter::create($this->object['contactInformation']['vimeoVideo'] ?? null);
        if ($vimeoVideo) {
            $media[] = $vimeoVideo;
        }

        return $media;
    }

    public function getMainMedium(): ?MediumAdapterInterface
    {
        $medium = $this->object['media'][0] ?? null;
        if (!$medium) {
            return null;
        }
        return new MediumAdapter($medium);
    }

    public function getFiles(): array
    {
        $files = array_map(
            function ($item) {
                return new FileAdapter($item);
            },
            $this->object['files'] ?? []
        );

        foreach ($this->object['tour']['gpsTracks'] ?? [] as $track) {
            $name = 'GPS-Track';
            if (strpos($track['file']['url'], '.gpx') !== false) {
                $name .= ' (GPX)';
            } elseif (strpos($track['file']['url'], '.kml') !== false) {
                $name .= ' (KML)';
            }

            $files[] = new FileAdapter($track['file'], $name);
        }

        return $files;
    }

    public function hasAttributes(): bool
    {
        return !empty($this->extractAttributes());
    }

    public function getAttributes(): array
    {
        return $this->extractAttributes();
    }

    public function getSourceName(): ?string
    {
        return $this->object['client']['name'] ?? null;
    }

    public function getAuthorName(): ?string
    {
        return $this->object['author'] ?? null;
    }

    public function getBookingUris(): array
    {
        return [];
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOpeningTimes(): ?string
    {
        $openingTimes = $this->object['openingTimes'] ?? null;
        if ($openingTimes) {
            return json_encode($openingTimes);
        }
        return null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        $openingTimes = $this->object['openingTimes'] ?? null;
        return $openingTimes ? 'toubizV1' : null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getExternalId(): string
    {
        return $this->object['id'];
    }

    private function extractAttributes(): array
    {
        $blueprints = [];
        $blueprintsToGroups = [];
        foreach ($this->object['fieldBlueprints'] as $group) {
            foreach ($group['fieldSets'] as $fieldSet) {
                foreach ($fieldSet['fields'] as $field) {
                    $blueprints[$field['id']] = $field;
                    $blueprintsToGroups[$field['id']] = $group['id'];
                }
            }
        }

        [ $mappedDynamicFields, $mappedFields ] = $this->extractMappedDynamicFields($blueprints, $blueprintsToGroups);
        $unmappedDynamicFields = $this->extractUnmappedAttributes($blueprints, $blueprintsToGroups, $mappedFields);
        return array_merge(
            $this->getFixedAttributes(),
            $this->getMappedRegular(),
            $mappedDynamicFields,
            $unmappedDynamicFields
        );
    }

    private function extractGeometry(): array
    {
        if (!$this->cachedGeometry) {
            $gpsTrackUrl = $this->object['tour']['gpsTracks'][0]['file']['url'] ?? null;
            $gpsTrackPoints = $this->object['tour']['gpsTracks'][0]['file']['points'] ?? null;
            if (!empty($gpsTrackPoints)) {
                $this->cachedGeometry = $gpsTrackPoints;
            } elseif (is_string($gpsTrackUrl)) {
                $track = (new ParserFactory(new XmlFileReader()))
                    ->resolveParser($gpsTrackUrl)
                    ->extractTrack($gpsTrackUrl);

                if ($track->isEmpty()) {
                    $this->logger->warning(sprintf(
                        'GPS File %s does not contain any track/point information.',
                        $gpsTrackUrl
                    ));
                }

                $this->cachedGeometry = $track->mapPoints(function (Point $point) {
                    return [ $point->getLatitude(), $point->getLongitude() ];
                });
            } else {
                $this->cachedGeometry = [];
            }
        }

        return $this->cachedGeometry;
    }

    private function getFixedAttributes(): array
    {
        $attributes = [
            $this->getGeometryAttribute(),
            $this->getPriceAttribute(),
            $this->getDurationAttribute(),
            $this->getDistanceAttribute(),
            $this->getTypeOfTourAttribute(),
            $this->getCapacityAttribute(),
        ];

        $attributes = array_merge(
            $attributes,
            $this->getRecommendedTimeOfTravelAttributes(),
            $this->getTourProperties()
        );
        return array_filter($attributes);
    }

    private function monthToConstant(string $month): ?string
    {
        switch ($month) {
            case 'january':
                return MonthConstants::JANUARY;
            case 'february':
                return MonthConstants::FEBRUARY;
            case 'march':
                return MonthConstants::MARCH;
            case 'april':
                return MonthConstants::APRIL;
            case 'may':
                return MonthConstants::MAY;
            case 'june':
                return MonthConstants::JUNE;
            case 'july':
                return MonthConstants::JULY;
            case 'august':
                return MonthConstants::AUGUST;
            case 'september':
                return MonthConstants::SEPTEMBER;
            case 'october':
                return MonthConstants::OCTOBER;
            case 'november':
                return MonthConstants::NOVEMBER;
            case 'december':
                return MonthConstants::DECEMBER;
            default:
                return null;
        }
    }

    private function getMappedRegular(): array
    {
        $attributes = [];
        foreach (AttributeMap::get() as [ $attributeName, $type, $path ]) {
            $attributeName = trim($attributeName);
            if ($type !== 'regular' || empty($attributeName)) {
                continue;
            }

            $value = ArrayUtility::arrayGet($this->object, explode('.', $path));
            if (is_string($value) || is_int($value) || is_bool($value)) {
                switch ($attributeName) {
                    // Toubiz backend uses a scale of 0 = easy, 1 = medium, 2 = hard
                    // while the frontend uses 1 = easy, 2 = medium, 3 = hard.
                    case TourAttributes::DIFFICULTY_RATING:
                        $value = ((int) $value) + 1;
                        break;
                }

                $attributes[] = new AttributeAdapter(
                    $attributeName,
                    $value,
                    null,
                    $this->getExternalId() . '__' . $attributeName
                );
            } elseif (is_array($value)) {
                foreach ($value as $index => $v) {
                    $attributes[] = new AttributeAdapter(
                        $attributeName,
                        $v,
                        null,
                        $this->getExternalId() . '__' . $attributeName . '__' . $index
                    );
                }
            }
        }

        return $attributes;
    }

    private function extractMappedDynamicFields(array $blueprints, array $blueprintsToGroups): array
    {
        $attributes = [];
        $fieldIds = [];

        $prefixedAwards = [
            GastronomyAttributes::HOUSE_DECORATION_MICHELIN,
            GastronomyAttributes::HOUSE_DECORATION_MILLAU,
            GastronomyAttributes::HOUSE_DECORATION_VARTA,
            GastronomyAttributes::HOUSE_DECORATION_ARAL,
        ];

        foreach (AttributeMap::get() as [ $attributeName, $type, $fieldId, $fieldValue, $attributeValue ]) {
            $attributeName = trim($attributeName);
            if (empty($attributeName)
                || strpos($attributeName, '#') === 0
                || !array_key_exists($fieldId, $blueprints)
                || $type !== 'dynamicField'
            ) {
                continue;
            }

            $i = 0;
            foreach ($this->object['fieldValues'][$fieldId] ?? [] as $dynamicFieldValue) {
                foreach (AttributeAdapter::extractValueSets($dynamicFieldValue) as $value => $humanReadableValue) {
                    if ($fieldValue && $value !== $fieldValue) {
                        continue;
                    }

                    $humanReadableValue = $attributeValue ?: $humanReadableValue;
                    if (strtolower($humanReadableValue) === 'true') {
                        $humanReadableValue = true;
                    } elseif (strtolower($humanReadableValue) === 'false') {
                        $humanReadableValue = false;
                    }

                    if (\in_array($attributeName, $prefixedAwards)) {
                        $humanReadableValue = sprintf('%s: %s', $blueprints[$fieldId]['name'], $humanReadableValue);
                    }


                    $fieldIds[] = $fieldId;
                    $attributes[] = new AttributeAdapter(
                        $attributeName,
                        $humanReadableValue,
                        $blueprintsToGroups[$fieldId],
                        sprintf('%s_%d', $blueprints[$fieldId]['id'], $i++)
                    );
                }
            }
        }

        return [ $attributes, array_unique($fieldIds) ];
    }

    private function extractUnmappedAttributes(array $blueprints, array $blueprintsToGroups, array $ignoreFields): array
    {
        $attributes = [];
        foreach ($blueprints as $id => $blueprint) {
            $i = 0;
            if (in_array($id, $ignoreFields, true) || !array_key_exists($id, $this->object['fieldValues'])) {
                continue;
            }

            foreach ($this->object['fieldValues'][$id] as $dynamicFieldValue) {
                foreach (AttributeAdapter::extractValueSets($dynamicFieldValue) as $value => $humanReadableValue) {
                    $attributes[] = new AttributeAdapter(
                        $blueprint['name'],
                        $humanReadableValue,
                        $blueprintsToGroups[$id],
                        sprintf('%s_%d', $id, $i++)
                    );
                }
            }
        }
        return $attributes;
    }

    /**
     * @inheritDoc
     */
    public function getCitySelectors(): array
    {
        return array_map(
            function (array $item) {
                return new ExternalIdSelector(ExternalIdType::TOUBIZ, $item['id']);
            },
            $this->object['relatedArticles']['areas'] ?? []
        );
    }

    /**
     * @inheritDoc
     */
    public function getAdditionalExternalIds(): array
    {
        $ids = [
            new ExternalIdAdapter(ExternalIdType::TOUBIZ, $this->getExternalId())
        ];

        $toubizLegacyRemoteId = null;
        $toubizLegacyId = null;
        foreach ($this->object['externalIds'] ?? [] as $service => $id) {
            [ $service, $mappedId ] = $this->mapExternalIdService($service, $id);
            $ids[] = new ExternalIdAdapter($service, $mappedId);

            if ($service === ExternalIdType::TOUBIZ_LEGACY) {
                $toubizLegacyId = $id;
            } elseif ($service === ExternalIdType::TOUBIZ_LEGACY_REMOTE_ID) {
                $toubizLegacyRemoteId = $id;
            }
        }

        if ($toubizLegacyId !== null && $toubizLegacyRemoteId === null) {
            $remoteId = $this->guessRemoteIdBasedOnPrefixedToubizLegacyId($toubizLegacyId);
            if ($remoteId) {
                $ids[] = new ExternalIdAdapter(ExternalIdType::TOUBIZ_LEGACY_REMOTE_ID, $remoteId);
            }
        }

        return $ids;
    }

    private function mapExternalIdService(string $service, string $id): array
    {
        switch ($service) {
            case 'toubizLegacy':
            case 'toubizId':
                $parts = explode('.', $id);
                $id = (string) array_pop($parts);
                return [ ExternalIdType::TOUBIZ_LEGACY, $id ];
            case 'toubizLegacyRemoteId':
                return [ ExternalIdType::TOUBIZ_LEGACY_REMOTE_ID, $id ];
            case 'outdoorActiveId':
                return [ ExternalIdType::OUTDOORACTIVE, $id ];
            case 'tomasId':
                return [ ExternalIdType::TOMAS, $id ];
            default:
                return [ $service, $id ];
        }
    }

    /**
     * Sometimes, no original id was saved on articles.
     * This method tries to guess the correct remote_id based on the prefixes in the toubiz id.
     *
     * @see https://newland.atlassian.net/wiki/spaces/INFOSYSTEM/pages/1524072466/External+IDs
     */
    private function guessRemoteIdBasedOnPrefixedToubizLegacyId(string $id): ?string
    {
        $parts = explode('.', $id);
        $id = (string) array_pop($parts);
        $prefix = implode('.', $parts);

        switch ($prefix) {
            case 'g':
                return 'tbgastro_' . $id;
            case 'd':
                return 'tbdirekt_' . $id;
            case 'city':
                return 'location_' . $id;
            case 'poi':
            case 'tour':
            default:
                // No common prefix exists
                return null;
        }
    }

    public function getFacebookUri(): ?string
    {
        return ($this->object['contactInformation']['facebook'] ?? null) ?: null;
    }

    public function getTwitterUri(): ?string
    {
        return ($this->object['contactInformation']['twitter'] ?? null) ?: null;
    }

    public function getInstagramUri(): ?string
    {
        return ($this->object['contactInformation']['instagram'] ?? null) ?: null;
    }

    public function getYoutubeUri(): ?string
    {
        return ($this->object['contactInformation']['youtube'] ?? null) ?: null;
    }

    public function getWikipediaUri(): ?string
    {
        return ($this->object['contactInformation']['wikipedia'] ?? null) ?: null;
    }

    public function getFlickrUri(): ?string
    {
        return ($this->object['contactInformation']['flickr'] ?? null) ?: null;
    }


    private function getGeometryAttribute(): ?AttributeAdapterInterface
    {
        $geometry = $this->extractGeometry();

        if (!empty($geometry)) {
            return new AttributeAdapter(
                TourAttributes::GEOMETRY,
                $geometry,
                null,
                $this->getExternalId() . '__geometry'
            );
        }

        return null;
    }

    private function getPriceAttribute(): ?AttributeAdapterInterface
    {
        $currencies = $this->object['point']['price']['currencies'] ?? [];
        $priceGroups = $this->object['point']['price']['priceGroups'] ?? [];
        if (empty($priceGroups)) {
            return null;
        }

        $table = [];
        foreach ($priceGroups as $priceGroup) {
            foreach ($priceGroup['priceEntries'] ?? [] as $priceEntry) {
                $row = [ $priceGroup['title'], $priceEntry['title'] ];
                foreach ($currencies as $currency) {
                    if ($currency === 'eur') {
                        $row[] = sprintf('%.2f €', $priceEntry['eur']);
                    } elseif ($currency === 'chf') {
                        $row[] = sprintf('CHF %.2f', $priceEntry['eur']);
                    } else {
                        $row[] = sprintf('%s %.2f', $currency, $priceEntry[$currency]);
                    }
                }
                $row[] = $priceEntry['comment'];
                $table[] = $row;
            }
        }

        return new AttributeAdapter(
            PointOfInterestAttributes::PRICES,
            $table,
            null,
            $this->getExternalId() . '__prices'
        );
    }

    private function getRecommendedTimeOfTravelAttributes(): array
    {
        $recommendedTimeOfTravel = $this->object['tour']['recommendedTimeOfTravel'] ?? null;
        if (!$recommendedTimeOfTravel) {
            return [ ];
        }

        $attributes = [];
        foreach ($recommendedTimeOfTravel as $month => $recommended) {
            if (!$recommended) {
                continue;
            }

            $attributes[] = new AttributeAdapter(
                TourAttributes::BEST_SEASON,
                $this->monthToConstant($month),
                null,
                $this->getExternalId() . '__best-season__' . $month
            );
        }
        return $attributes;
    }

    private function getDurationAttribute(): ?AttributeAdapterInterface
    {
        $duration = $this->object['tour']['combinedTrackInformation']['duration'] ?? null;
        if ($duration) {
            return new AttributeAdapter(
                TourAttributes::TOUR_DURATION,
                $duration,
                null,
                $this->getExternalId() . '__duration'
            );
        }

        // TODO Remove old hours handling
        $hours = $this->object['tour']['combinedTrackInformation']['hours'] ?? null;
        if ($hours) {
            return new AttributeAdapter(
                TourAttributes::TOUR_DURATION,
                $hours * 60,
                null,
                $this->getExternalId() . '__duration'
            );
        }

        return null;
    }

    private function getDistanceAttribute(): ?AttributeAdapterInterface
    {
        $distance = $this->object['tour']['combinedTrackInformation']['distance'] ?? null;
        if ($distance) {
            return new AttributeAdapter(
                TourAttributes::TOUR_LENGTH,
                $distance,
                null,
                $this->getExternalId() . '__distance'
            );
        }

        // TODO Remove old kilometers handling
        $kilometers = $this->object['tour']['combinedTrackInformation']['kilometers'] ?? null;
        if ($kilometers) {
            return new AttributeAdapter(
                TourAttributes::TOUR_LENGTH,
                $kilometers * 1000,
                null,
                $this->getExternalId() . '__distance'
            );
        }

        return null;
    }

    private function getCapacityAttribute(): ?AttributeAdapterInterface
    {
        $dynamicFieldIds = [
            'inside' => '01480042-3e62-47c6-b3e6-4af6cc7e18f0',
            'outside' => '36ae8e69-5ccb-4624-bf20-8f7f088549b4',
            'sideRooms' => '4244b0d0-92d8-4161-ad34-91ba16ab3db5',
        ];

        $data = [];
        foreach ($dynamicFieldIds as $fieldName => $fieldId) {
            $data[$fieldName] = $this->object['fieldValues'][$fieldId][0]['humanReadableValue']['text'] ?? null;
        }

        $formats = [
            'inside' => [
                'de' => 'Innen: %s',
                'en' => 'inside: %s',
            ],
            'outside' => [
                'de' => 'Außen: %s',
                'en' => 'outside: %s',
            ],
            'sideRooms' => [
                'de' => 'In Nebenräumen: %s',
                'en' => 'In siderooms: %s',
            ]
        ];

        $formatted = [];
        foreach ($data as $field => $value) {
            if ($value) {
                $formatted[] = sprintf($formats[$field][$this->language] ?? $formats[$field]['en'], $value);
            }
        }

        if (empty($formatted)) {
            return null;
        }

        return new AttributeAdapter(
            GastronomyAttributes::CAPACITY,
            implode(', ', $formatted),
            null,
            $this->object['id'] . '__capacity'
        );
    }

    /** @return AttributeAdapterInterface[] */
    private function getTourProperties(): array
    {
        $propertyKeys = [
            'scenic' => [
                'de' => 'Aussichtsreicht',
                'en' => 'Scenic',
            ],
            'geology' => [
                'de' => 'Geologische Highlights',
                'en' => 'Geological Highlights',
            ],
            'flora' => [
                'de' => 'Botanische Highlights',
                'en' => 'Botanic Highlights',
            ],
            'culture' => [
                'de' => 'Kulturelle Highlights',
                'en' => 'Cultural Highlights',
            ],
            'fauna' => [
                'de' => 'Faunistische Highlights',
                'en' => 'Fauna Highlights',
            ],
            'healthyClimate' => [
                'de' => 'Heilklimatisch',
                'en' => 'Healthy Climate',
            ],
        ];

        $properties = [];
        foreach ($propertyKeys as $key => $labels) {
            if ($this->object['tour'][$key] ?? false) {
                $properties[] = new AttributeAdapter(
                    TourAttributes::PROPERTIES,
                    $labels[$this->language] ?? $labels['en'],
                    null,
                    sprintf('%s__%s', $this->object['id'], $key)
                );
            }
        }

        return $properties;
    }

    private function getTypeOfTourAttribute(): ?AttributeAdapterInterface
    {

        $typeOfTour = $this->object['tour']['typeOfTour'] ?? null;
        $mappedTypeOfTour = null;
        switch ($typeOfTour) {
            case 'loop_tour':
                $mappedTypeOfTour = 'loopTour';
                break;
            case 'one_way_tour':
                $mappedTypeOfTour = 'oneWayTour';
                break;
            case 'stage':
                $mappedTypeOfTour = 'stage';
                break;
            case 'stages_tour':
                $mappedTypeOfTour = 'multiStageTour';
                break;
            case 'mountaintop_tour':
                $mappedTypeOfTour = 'mountainTopTour';
                break;
            case 'multi_day_tour':
                $mappedTypeOfTour = 'multiDayTour';
                break;
        }
        if ($mappedTypeOfTour) {
            return new AttributeAdapter(
                TourAttributes::PROPERTIES,
                $mappedTypeOfTour,
                null,
                $this->getExternalId() . '__type-of-tour'
            );
        }

        return null;
    }
}
