<?php
namespace Newland\Toubiz\Api\Service\Tportal;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\LodgingAdapter;

/**
 * Service for legacy Toubiz TPortal.
 *
 * Concrete implementation to communicate with the TPortal
 * which is providing data for TOMAS-bound entities.
 */
class ApiService extends AbstractService
{
    /**
     * @var string Base URI of API endpoint.
     */
    const BASE_URI = 'https://tportal.toubiz.de';

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [
        'Lodging' => LodgingAdapter::class
    ];

    /**
     * Fetch lodgings.
     *
     * @param callable $block
     * @return void
     */
    public function fetchLodgings(callable $block)
    {
        $page = 1;
        while (true) {
            $data = $this->sendRequest($page);
            if (!$data) {
                return;
            }

            foreach ($data['housedata'] as $id => $item) {
                $item = $this->buildItemArray($id, $data);
                $lodging = new LodgingAdapter($item);
                $block($lodging);
            }

            $page++;
        }
    }

    /**
     * Builds an array of data for one lodging from given full array.
     *
     * @param string $id
     * @param array $data
     * @return array
     */
    protected function buildItemArray($id, $data)
    {
        return [
            'housedata' => $data['housedata'][$id],
            'searchresult_details' => $data['searchresult_details'][$id]
        ];
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $page
     * @return array|bool
     */
    protected function sendRequest($page)
    {
        $parameters = [
            ':clientName' => $this->clientName,
            ':page' => $page
        ];

        $url = str_replace(
            array_keys($parameters),
            array_values($parameters),
            '/:clientName/ukv/search?reset=1&json=1&ukv_result_order=3&page=:page'
        );

        $response = $this->httpClient->request('GET', $url);
        if ($response->getStatusCode() == 200) {
            $data = json_decode($response->getBody(), true);

            /*
             * Checking on housedata if data is present but this could
             * also be any other field that is being returned in the
             * array, as data is spread across multiple response segments.
             */
            if ($data['housedata'] && count($data['housedata']) > 0) {
                return $data;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }
}
