<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;

/**
 * Gastronomy adapter.
 *
 * This represents an Article with mapping for the Tportal-specific gastronomy.
 */
class GastronomyAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    /**
     * @return int
     */
    public function getMainType()
    {
        return ArticleConstants::TYPE_GASTRONOMY;
    }

    /**
     * @return string
     */
    public function getExternalId()
    {
        return $this->object['remote_id'];
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->object['name'];
    }

    /**
     * @return string|null
     */
    public function getAbstract()
    {
        return;
    }

    /**
     * @return string|null
     */
    public function getDescription()
    {
        if(!is_array($this->object['description'])) {
            return;
        }
        if (array_key_exists('gastro', $this->object['description'])) {
            $value = $this->object['description']['gastro']['text'];
        } elseif (array_key_exists('gastro_event', $this->object['description'])) {
            $value = $this->object['description']['gastro_event']['text'];
        }

        if (!empty($value)) {
            return $value;
        }
    }

    /**
     * @return string
     */
    public function getStreet()
    {
        return $this->object['address']['street'];
    }

    /**
     * @return string
     */
    public function getZip()
    {
        return $this->object['address']['zip'];
    }

    /**
     * @return string
     */
    public function getCity()
    {
        return $this->object['address']['city'];
    }

    /**
     * @return string|null
     */
    public function getCountry()
    {
        return null;
    }

    /**
     * @return float
     */
    public function getLatitude()
    {
        return $this->object['location']['Latitude'];
    }

    /**
     * @return float
     */
    public function getLongitude()
    {
        return $this->object['location']['Longitude'];
    }

    /**
     * @return array<AddressAdapter>
     */
    public function getAddresses()
    {
        $address = new AddressAdapter($this->object);
        return [$address];
    }

    /**
     * @return array<CategoryAdapter>
     */
    public function getCategories()
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [$category];
    }

    /**
     * @return array<MediumAdapter>
     */
    public function getMedia()
    {
        if (!is_array($this->object['images'])
            || !array_key_exists('all', $this->object['images'])) {
            return;
        }

        $media = [];
        foreach ($this->object['images']['all'] as $image) {
            $media[] = (new MediumAdapter($image));
        }
        return $media;
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        $attributes = [];
        $keys = [
            'gastro_ambient' => 'ambiance',
            'gastro_child' => 'families',
            'gastro_group' => 'groups',
            'gastro_kitchen' => 'kitchenStyle',
            'gastro_style' => 'gastronomyStyle',
            'gastro_person' => 'capacity',
            'gastro_payment' => 'acceptedPayments',
            'gastro_pricesegment' => 'priceSegment',
            'gastro_property' => 'services',
            'gastro_language' => 'spokenLanguages'
        ];

        foreach ($keys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key);
        }

        return $attributes;
    }

    /**
     * @return string
     */
    public function getFacebookUri()
    {
        if(!is_array($this->object['social_media'])) {
            return;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'facebook'
        );
        if ($data) {
            return $data['value'];
        }
    }

    /**
     * @return string
     */
    public function getTwitterUri()
    {
        if(!is_array($this->object['social_media'])) {
            return;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'twitter'
        );
        if ($data) {
            return $data['value'];
        }
    }

    /**
     * @return string
     */
    public function getInstagramUri()
    {
        if(!is_array($this->object['social_media'])) {
            return;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'instagram'
        );
        if ($data) {
            return $data['value'];
        }
    }

    /**
     * @return string
     */
    public function getYoutubeUri()
    {
        if(!is_array($this->object['social_media'])) {
            return;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'youtube'
        );
        if ($data) {
            return $data['value'];
        }
    }

    /**
     * @return array<FileAdapter>
     */
    public function getFiles()
    {
        if (!is_array($this->object['files'])) {
            return;
        }

        $files = [];
        foreach ($this->object['files'] as $file) {
            $files[] = (new FileAdapter($file));
        }
        return $files;
    }

    /**
     * @return string
     */
    public function getSourceName()
    {
        return $this->object['headline_quellmandant'];
    }

    /**
     * @return string|null
     */
    public function getAuthorName()
    {
        if (!empty($this->object['author'])) {
            return $this->object['author'];
        }
    }

    /**
     * @return string
     */
    public function getBookingUri()
    {
        if (!is_array($this->object['property'])
            || !array_key_exists('gastro_lunchgate', $this->object['property'])
            || !is_array($this->object['property']['gastro_lunchgate'])) {
            return;
        }

        $value = $this->object['property']['gastro_lunchgate'][0]['properties'][0]['value'];

        // This is wrong data due to missing valitations.
        if (stristr($value, '@')) {
            return;
        }

        return $value;
    }

    /**
     * Extract attributes from a volatile set of data.
     *
     * @param string $key
     * @return array|null
     */
    protected function extractAttributes($key)
    {
        $items = [];

        // The desired key can be in two different places.
        if (array_key_exists($key, $this->object)) {
            $attributes = $this->object[$key];
        } elseif (is_array($this->object['property'])
                  && array_key_exists($key, $this->object['property'])
        ) {
            $attributes = $this->object['property'][$key];
        } else {
            return null;
        }

        // Also, the structure is different for finding the attributes.
        if (is_array($attributes[0])
            && array_key_exists('properties', $attributes[0])
        ) {
            $attributes = $attributes[0]['properties'];
        }

        foreach ($attributes as $entry) {
            /*
             * Most of the time, text is only used. But if it also
             * has a value given, it needs to be concatenated.
             */
            if (!empty($entry['value'])) {
                $items[] = $entry['text'] . ': ' . $entry['value'];
            } else {
                $items[] = $entry['text'];
            }
        }

        if (!empty($items)) {
            return $items;
        }
    }
}
