<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\GastronomyAdapter;

/**
 * Service for legacy Toubiz gastronomy API.
 *
 * Concrete implementation to communicate with
 * the first, "old" toubiz API returning gastronomy articles.
 */
class GastronomyApiService extends AbstractService
{
    use LanguageAware;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://api.toubiz.de/json/gastro/';

    /**
     * Map of language constants to URI segments expected by the API.
     *
     * @var array
     */
    private static $languageMap = [
        Language::DE => 'de',
        Language::EN => 'en',
        Language::FR => 'fr',
        Language::ES => 'es',
    ];

    public function fetchGastronomy(callable $block): ServiceResult
    {
        $deleted = [ [] ];
        $concurrency = $this->parameters['concurrency'] ?? 10;
        $perPage = $this->parameters['perPage'] ?? 100;

        $pool = new ConcurrentPaginatedRequests($concurrency, function (int $page) use ($perPage, $block, &$deleted) {
            return $this->sendRequest($page, $perPage)->then(function (array $data) use ($block, &$deleted) {
                $deleted[] = $data['deleted'] ?? [];
                foreach ($data['result']['item'] ?? [] as $item) {
                    $block($this->gastronomyDataToAdapter($item), $data['result']['itemCount'] ?? null);
                }
            });
        });

        $pool->start()->wait();

        $result = new ServiceResult();
        $result->setDeleted(array_map(
            function ($id) {
                return 'tbgastro_' . $id;
            },
            array_merge(...$deleted)
        ));

        return $result;
    }

    /**
     * Requests data from the API.
     *
     * @param int $page
     * @param int $limit
     * @return PromiseInterface
     */
    protected function sendRequest(int $page, int $limit): PromiseInterface
    {
        $offset = ($page - 1) * $limit;
        $uri = Uri::withQueryValues(new Uri('modified.php'), [
            'key' => $this->apiKey,
            'modified' => $this->delta->asSeconds(),
            'offset' => $offset,
            'limit' => $limit,
            'language' => (static::$languageMap[$this->language] ?? 'de'),
        ]);

        return $this->jsonRequest($uri, 5, function (array $data) {
            return !empty($data['result']);
        });
    }

    private function gastronomyDataToAdapter(array $item): GastronomyAdapter
    {
        $article = new GastronomyAdapter($item);
        if ($this->language) {
            $article->setLanguage($this->language);
        }
        return $article;
    }
}
