<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\ObjectAdapter;

use Newland\Toubiz\Api\ObjectAdapter\Concern\MediumConstants;
use Newland\Toubiz\Api\Exception\UnknownVideoException;

class VimeoVideoAdapter implements MediumAdapterInterface, MediumWithGeneratedPreviewUri
{
    public static function create(?string $url): ?VimeoVideoAdapter
    {
        if ($url === null) {
            return null;
        }
        $videoId = static::extractVideoId($url);
        return $videoId ? new VimeoVideoAdapter($url, $videoId) : null;
    }

    /** @var string */
    private $videoId;

    /** @var string */
    private $videoUrl;

    public function __construct(string $url, string $videoId)
    {
        $this->videoUrl = $url;
        $this->videoId = $videoId;
    }

    public function getExternalId(): string
    {
        return 'vimeo_' . $this->videoId;
    }

    public function getSourceUri(): string
    {
        return $this->videoUrl;
    }

    public function getTitle(): ?string
    {
        return null;
    }

    public function getCopyright(): ?string
    {
        return null;
    }

    public function getType(): string
    {
        return MediumConstants::TYPE_VIDEO;
    }


    protected static function extractVideoId(string $url): ?string
    {
        $parsedUrl = parse_url($url);
        if ($parsedUrl === false) {
            throw new UnknownVideoException('The url ' . $url . ' cannot be parsed.');
        }

        if (array_key_exists('path', $parsedUrl)) {
            $pathParts = explode('/', $parsedUrl['path']);
            return (string) array_pop($pathParts);
        }

        throw new UnknownVideoException('The url ' . $url . ' cannot be parsed as Vimeo URL.');
    }

    public function generatePreviewUri(): ?string
    {
        $url = sprintf(
            'https://vimeo.com/api/oembed.json?width=640&fun=false&url=%s',
            urlencode($this->videoUrl)
        );
        $data = json_decode((string) file_get_contents($url), true);
        return $data['thumbnail_url'] ?? null;
    }
}
