<?php
namespace Newland\ToubizApi\Tests\Service\Toubiz\Legacy\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\GastronomyAdapter;
use PHPUnit\Framework\TestCase;

class GastronomyAdapterTest extends TestCase
{


    const JSON_DATA = '/../../../../../../Fixtures/GastroData/gastroData.json';

    /** @dataProvider provideRealWorldApiResponseData */
    public function testMediumImport(array $data): void
    {
        $adapter = new GastronomyAdapter($data);
        $this->assertGreaterThanOrEqual(1, $adapter->getMedia());
    }

    /** @dataProvider provideRealWorldApiResponseData */
    public function testLatitudeAndLongitudeImport(array $data): void
    {
        $adapter = new GastronomyAdapter($data);
        $this->assertIsFloat($adapter->getLongitude());
        $this->assertIsFloat($adapter->getLatitude());
    }

    /** @dataProvider provideRealWorldApiResponseData */
    public function testProvidesArrayOfAtLeastOneCategoryAdapter(array $data): void
    {
        $adapter = new GastronomyAdapter($data);
        $this->assertGreaterThanOrEqual(1, \count($adapter->getCategories()));
        foreach ($adapter->getCategories() as $category) {
            $this->assertInstanceOf(CategoryAdapterInterface::class, $category);
        }
    }

    public function testMapsMultipleCategories(): void
    {
        $adapter = new GastronomyAdapter(
            [
                'remote_id' => 'test',
                'category' => [
                    [
                        'property' => '66a40a311003cdacc1a4f8b8e4d108d7',
                        'group' => 'house_category_gastro',
                        'text' => 'Biergarten',
                        'value' => '',
                    ],
                    [
                        'property' => 'c7841959f9aa5bcda0b23961242e4408',
                        'group' => 'house_category_gastro',
                        'text' => "Gasthaus \/ Gasthof \/ Gastst\u00e4tte",
                        'value' => '',
                    ],
                    [
                        'property' => '61037d2ebe8a2b718f50d816615b0ca2',
                        'group' => 'house_category_gastro',
                        'text' => 'Restaurant',
                        'value' => '',
                    ],
                    [
                        'property' => '90e3be04d8f76b1f77eb6a2958263cae',
                        'group' => 'house_category_gastro',
                        'text' => "\u00dcbernachtungsm\u00f6glichkeit",
                        'value' => '',
                    ],
                ],
            ]
        );

        $this->assertCount(4, $adapter->getCategories());
        $this->assertEquals('66a40a311003cdacc1a4f8b8e4d108d7', $adapter->getCategories()[0]->getExternalId());
        $this->assertEquals('c7841959f9aa5bcda0b23961242e4408', $adapter->getCategories()[1]->getExternalId());
        $this->assertEquals('61037d2ebe8a2b718f50d816615b0ca2', $adapter->getCategories()[2]->getExternalId());
        $this->assertEquals('90e3be04d8f76b1f77eb6a2958263cae', $adapter->getCategories()[3]->getExternalId());
    }

    public function testMainCategoryIsMappedIfThereAreNoAdditionalCategories(): void
    {
        $adapter = new GastronomyAdapter(
            [
                'main_category' => [
                    'property' => '66a40a311003cdacc1a4f8b8e4d108d7',
                    'group' => 'house_category_gastro',
                    'text' => 'Biergarten',
                    'value' => '',
                ],
            ]
        );

        $this->assertCount(1, $adapter->getCategories());
        $this->assertEquals('66a40a311003cdacc1a4f8b8e4d108d7', $adapter->getCategories()[0]->getExternalId());
    }

    public function testMainCategoryIsAlsoMappedAsCategory(): void
    {
        $adapter = new GastronomyAdapter(
            [
                'main_category' => [
                    'property' => '66a40a311003cdacc1a4f8b8e4d108d7',
                    'group' => 'house_category_gastro',
                    'text' => 'Biergarten',
                    'value' => '',
                ],
                'category' => [
                    [
                        'property' => '61037d2ebe8a2b718f50d816615b0ca2',
                        'group' => 'house_category_gastro',
                        'text' => 'Restaurant',
                        'value' => '',
                    ],
                ],
            ]
        );


        $this->assertCount(2, $adapter->getCategories());
        $this->assertEquals('66a40a311003cdacc1a4f8b8e4d108d7', $adapter->getCategories()[0]->getExternalId());
        $this->assertEquals('61037d2ebe8a2b718f50d816615b0ca2', $adapter->getCategories()[1]->getExternalId());
    }

    public function testMainCategoryIsNotInCategoriesTwiceIfAlsoInCategoriesArray(): void
    {
        $adapter = new GastronomyAdapter(
            [
                'main_category' => [
                    'property' => '66a40a311003cdacc1a4f8b8e4d108d7',
                    'group' => 'house_category_gastro',
                    'text' => 'Biergarten',
                    'value' => '',
                ],
                'category' => [
                    [
                        'property' => '66a40a311003cdacc1a4f8b8e4d108d7',
                        'group' => 'house_category_gastro',
                        'text' => 'Biergarten',
                        'value' => '',
                    ],
                    [
                        'property' => '61037d2ebe8a2b718f50d816615b0ca2',
                        'group' => 'house_category_gastro',
                        'text' => 'Restaurant',
                        'value' => '',
                    ],
                ],
            ]
        );

        $this->assertCount(2, $adapter->getCategories());
        $this->assertEquals('66a40a311003cdacc1a4f8b8e4d108d7', $adapter->getCategories()[0]->getExternalId());
        $this->assertEquals('61037d2ebe8a2b718f50d816615b0ca2', $adapter->getCategories()[1]->getExternalId());
    }

    public function testMainCategoryIsAlwaysSortedUp(): void
    {

        $adapter = new GastronomyAdapter(
            [
                'main_category' => [
                    'property' => '66a40a311003cdacc1a4f8b8e4d108d7',
                    'group' => 'house_category_gastro',
                    'text' => 'Biergarten',
                    'value' => '',
                ],
                'category' => [
                    [
                        'property' => '61037d2ebe8a2b718f50d816615b0ca2',
                        'group' => 'house_category_gastro',
                        'text' => 'Restaurant',
                        'value' => '',
                    ],
                    [
                        'property' => '66a40a311003cdacc1a4f8b8e4d108d7',
                        'group' => 'house_category_gastro',
                        'text' => 'Biergarten',
                        'value' => '',
                    ],
                ],
            ]
        );

        $this->assertCount(2, $adapter->getCategories());
        $this->assertEquals('66a40a311003cdacc1a4f8b8e4d108d7', $adapter->getCategories()[0]->getExternalId());
        $this->assertEquals('61037d2ebe8a2b718f50d816615b0ca2', $adapter->getCategories()[1]->getExternalId());
    }

    /** @dataProvider provideProperties */
    public function testExtractsMultiplePropertiesAsMultipleAttributes(
        string $propertyName,
        int $count,
        $propertyValue
    ): void {
        $adapter = new GastronomyAdapter(
            [
                'remote_id' => 'test',
                'property' => [
                    $propertyName => $propertyValue,
                ],
            ]
        );

        $this->assertCount($count, $adapter->getAttributes());
    }

    public function provideProperties(): array
    {
        return [
            'simple' => [
                'gastro_kitchen',
                2,
                [
                    [ 'text' => 'foo' ],
                    [ 'text' => 'bar' ],
                ],
            ],
            'properties' => [
                'gastro_kitchen',
                2,
                [
                    [
                        'properties' => [
                            [ 'text' => 'foo' ],
                            [ 'text' => 'bar' ],
                        ],
                    ],
                ],
            ],
            'multiple properties' => [
                'gastro_kitchen',
                3,
                [
                    [ 'properties' => [ [ 'text' => 'foo' ] ] ],
                    [ 'properties' => [ [ 'text' => 'bar' ], [ 'text' => 'baz' ] ] ],
                ],
            ],
        ];
    }

    public function testPrependsValueIfAvailable(): void
    {
        $adapter = new GastronomyAdapter(
            [
                'remote_id' => 'test',
                'property' => [
                    'gastro_kitchen' => [
                        [ 'value' => 'foo', 'text' => 'bar' ],
                    ],
                ],
            ]
        );

        $this->assertEquals('bar: foo', $adapter->getAttributes()[0]->getData());
    }

    /** @dataProvider provideAwardNames */
    public function testMapsAwardsCorrectly(string $normalizedAwardName, string $toubizName): void
    {
        $label = sprintf('___LABEL___%s___', md5(random_bytes(32)));
        $value = sprintf('___VALUE___%s___', md5(random_bytes(32)));

        $adapter = new GastronomyAdapter(
            [
                'remote_id' => 'test',
                'decoration' => [
                    $toubizName => [
                        [
                            'name' => $toubizName,
                            'properties' => [
                                [
                                    'property' => md5(random_bytes(32)),
                                    'group' => $toubizName,
                                    'text' => $label . '###' . md5(random_bytes(32)),
                                    'value' => $value . '###',
                                ],
                            ],
                        ],
                    ],
                ],
            ]
        );

        $this->assertEquals($normalizedAwardName, $adapter->getAttributes()[0]->getName());
        $this->assertEquals($label, $adapter->getAttributes()[0]->getData()['label']);
        $this->assertEquals($value, $adapter->getAttributes()[0]->getData()['value']);
    }

    /**
     * @dataProvider getAwardEdgeCases
     */
    public function testAwardDataEdgeCases(
        string $labelInput,
        string $valueInput,
        string $labelOutput,
        string $valueOutput,
        string $descriptionOutput
    ): void {
        $adapter = new GastronomyAdapter(
            [
                'remote_id' => 'test',
                'decoration' => [
                    'house_decoration_varta' => [
                        [
                            'name' => 'house_decoration_varta',
                            'properties' => [
                                [
                                    'property' => md5(random_bytes(32)),
                                    'group' => 'house_decoration_varta',
                                    'text' => $labelInput,
                                    'value' => $valueInput,
                                ],
                            ],
                        ],
                    ],
                ],
            ]
        );

        $this->assertEquals($labelOutput, $adapter->getAttributes()[0]->getData()['label']);
        $this->assertEquals($valueOutput, $adapter->getAttributes()[0]->getData()['value']);
        $this->assertEquals($descriptionOutput, $adapter->getAttributes()[0]->getData()['description']);
    }

    public function provideRealWorldApiResponseData(): array
    {
        $path = realpath(__DIR__ . '/../../../../../../Fixtures/GastroData/2019-12-13__example-api-response.json');
        $data = json_decode(file_get_contents($path), true);
        return array_map(
            function ($item) {
                return [ $item ];
            },
            $data['result']['item']
        );
    }


    public function provideAwardNames(): array
    {
        return [
            [ 'houseAwardsGeneral', 'house_decoration' ],
            [ 'houseAwardsAral', 'house_decoration_aral' ],
            [ 'houseAwardsDehoga', 'house_decoration_dehoga' ],
            [ 'houseAwardsFine', 'house_decoration_fine' ],
            [ 'houseAwardsMichelin', 'house_decoration_michelin' ],
            [ 'houseAwardsMillau', 'house_decoration_millau' ],
            [ 'houseAwardsVarta', 'house_decoration_varta' ],
        ];
    }

    public function testIgnoresMediaWithEmptySourceUri(): void
    {
        $adapter = new GastronomyAdapter(
            [
                'remote_id' => 'test',
                'images' => [
                    [ 'id_image' => 'test', 'imageURL' => '' ],
                    [ 'id_image' => 'test', 'imageURL' => 'https://placehold.it/500x500' ],
                ],
            ]
        );

        $this->assertCount(1, $adapter->getMedia());
    }

    public function getAwardEdgeCases(): array
    {
        return [
            'withLineBreaks' => [
                '4 Diamanten###Varta-Diamanten sind Empfehlungen von Hotels und Restaurants, empfohlen in
einem Varta-Führer.<br>
<a href="http://www.varta-guide.de" target="_new">www.varta-guide.de </a>',
                'fourstar###',
                '4 Diamanten',
                'fourstar',
                "Varta-Diamanten sind Empfehlungen von Hotels und Restaurants, empfohlen in\neinem Varta-Führer.\nwww.varta-guide.de",
            ],
            'valueStringWithModifier' => [
                'foo_label###123456',
                'foo_value###16',
                'foo_label',
                'foo_value_16',
                '123456',
            ],
        ];
    }

    public function testMapsSearchString(): void
    {
        // real world data
        $adapter = new GastronomyAdapter([
            'remote_id' => 'tbgastro_26640',
            'searchwords' => [
                [ 'property' => null, 'group' => 'Searchword', 'text' => 'test', 'value' => null ],
                [ 'property' => null, 'group' => 'Searchword', 'text' => 'Gastronomie', 'value' => null ],
            ]
        ]);

        $attributes = $adapter->getAttributes();
        $this->assertCount(1, $attributes);
        $this->assertEquals('additionalSearchString', $attributes[0]->getName());
        $this->assertEquals('test Gastronomie', $attributes[0]->getData());
    }

    public function testDoesNotMapSearchStringsWhenNonExistent(): void
    {
        // real world data
        $adapter = new GastronomyAdapter([ 'searchwords' => null ]);
        $attributes = $adapter->getAttributes();
        $this->assertCount(0, $attributes);
    }
    public function testDoesNotMapSearchStringsWhenEmpty(): void
    {
        // real world data
        $adapter = new GastronomyAdapter([ 'searchwords' => [] ]);
        $attributes = $adapter->getAttributes();
        $this->assertCount(0, $attributes);
    }
}
