<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Tests\Unit\Service\OutdoorActive\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Tests\UnitTestCase;
use Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter\TourAdapter;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;

class TourAdapterTest extends UnitTestCase
{
    /**
     * @var TourAdapter
     */
    protected $subject;

    public function setUp()
    {
        $data = json_decode(file_get_contents(__DIR__ . '/../../../../Fixtures/OutdoorActiveData/tour13025479.json'), true);
        $this->subject = new TourAdapter($data['tour'][0]);
    }

    public function testFirstImageIsNotReplacedByPrimaryImage(): void
    {
        $media = $this->subject->getMedia();
        $this->assertEquals('outdooractive_8581583', $media[0]->getExternalId());
    }

    public function testPrimaryImageIsNotFirstImage(): void
    {
        $this->assertEquals('outdooractive_8581637', $this->subject->getMainMedium()->getExternalId());
    }

    /** @dataProvider provideDescriptions */
    public function testProperlyStripDescription(string $expected, string $input): void
    {
        $adapter = new TourAdapter([ 'longText' => $input ]);
        $this->assertEquals($expected, $adapter->getDescription());
    }

    public function provideDescriptions(): array
    {
        return [
            'valid text remains text' => ['text', 'text'],
            'invalid script is completely discarded' => ['', '<script>text</script>'],
            'invalid tag is removed' => ['text', '<div>text</div>'],
            'invalid custom tag is removed' => ['text', '<myTag>text</myTag>'],
            'invalid empty tag' => ['', '<u></u>'],
            'valid code stays unchanged' => ['<u>text</u>', '<u>text</u>'],
            'invalid post spacing is removed' => ['<u>text</u>', '<u >text</u>'],
            'invalid data attribute' => ['<u>text</u>', '<u data>text</u>'],
            'invalid data attribute with value' => ['<u>text</u>', '<u data="foo">text</u>'],
            'invalid attribute for tag (valid uri)' =>
                ['<u>text</u>', '<u href="www.example.com">text</u>'],
            'invalid data attribute followed by invalid attribute for tag' =>
                ['<u>text</u>', '<u href="www.example.com" data>text</u>'],
            'invalid attribute for tag followed by invalid data attribute' =>
                ['<u>text</u>', '<u data href="www.example.com">text</u>'],
            'invalid attribute for tag (title)' =>
                ['<u>text</u>', '<u title="Title">text</u>'],
            'invalid attribute for tag (target)' =>
                ['<u>text</u>', '<u target="_blank">text</u>'],
            'invalid a tag without attribute href' =>
                ['text', '<a>text</a>'],
            'valid a tag with href' =>
                ['<a href="www.example.com">text</a>', '<a href="www.example.com">text</a>'],
            'invalid a tag using single quotes' =>
                ['<a href="www.example.com">text</a>', '<a href=\'www.example.com\'>text</a>'],
            'invalid duplicate href' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" href="www.example-2.com">text</a>'],
            'invalid duplicate href with invalid data attribute in between' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" data href="www.example-2.com">text</a>'],
            'valid a tag with title' =>
                ['<a href="www.example.com" title="Title">text</a>',
                    '<a href="www.example.com" title="Title">text</a>'],
            'valid a tag with target' =>
                ['<a href="www.example.com" target="_blank" rel="noreferrer noopener">text</a>',
                    '<a href="www.example.com" target="_blank">text</a>'],
            'valid nesting is kept' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</u></strong>'],
            'invalid nesting (wrong order) is corrected' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</strong></u>'],
            'invalid nesting (missing closing tags) are added in the end' =>
                ['<strong><u><em>text</em></u></strong>', '<strong><u><em>text</em></strong>'],
            'invalid nesting (only closing tag) is removed' =>
                ['<strong><em>text</em></strong>', '<strong><em>text</em></u></strong>'],
        ];
    }

    public function testProvidesCityRelationsBasedOnCommunes(): void
    {
        $adapter = new TourAdapter([
            'regions' => [
                'region' => [
                    [ 'type' => 'commune', 'id' => '123' ],
                    [ 'type' => 'country', 'id' => '333' ],
                    [ 'type' => 'province', 'id' => '333' ],
                    [ 'type' => 'mountainarea', 'id' => '333' ],
                    [ 'type' => 'protectedarea', 'id' => '333' ],
                    [ 'type' => 'commune', 'id' => '456' ],
                    [ 'type' => 'customarea', 'id' => '333' ],
                    [ 'type' => 'tourismarea', 'id' => '333' ],
                ]
            ]
        ]);

        $this->assertCount(2, $adapter->getCitySelectors());
        $this->assertEquals(ExternalIdType::OUTDOORACTIVE, $adapter->getCitySelectors()[0]->getType());
        $this->assertEquals('123', $adapter->getCitySelectors()[0]->getId());
        $this->assertEquals(ExternalIdType::OUTDOORACTIVE, $adapter->getCitySelectors()[1]->getType());
        $this->assertEquals('456', $adapter->getCitySelectors()[1]->getId());
    }
}
