<?php
namespace Newland\Toubiz\Api\Service\Outdooractive;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Promise\EachPromise;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Exception\InvalidServiceResponseException;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter\TourAdapter;
use Newland\Toubiz\Api\Service\ServiceResult;

/**
 * Service for Outdooractive API.
 */
class ApiService extends AbstractService
{
    use LanguageAware,
        Limitable;

    const DEFAULT_BASE_URI = 'https://www.outdooractive.com/api/project/';

    /**
     * `lastModifiedAfter` is added so the `lastModified` attribute is added to the response.
     * It is not however set to the current delta: That is resolved in the `fetchIdListWithinCurrentDelta`
     * method.
     */
    protected const PATH_TOURS_LIST = ':clientName/tours?key=:apiKey&lang=:language&lastModifiedAfter=1970-01-01';
    protected const PATH_TOUR_DETAIL = ':clientName/oois/:ids?key=:apiKey&lang=:language';
    protected const RECORDS_PER_DETAIL_REQUEST = 10;

    public function fetchTours(callable $block): ServiceResult
    {
        $all = $this->fetchTourList();
        if ($all === null) {
            throw new InvalidServiceResponseException('Could not fetch list of all tours');
        }

        if ($this->limit) {
            $all = array_splice($all, 0, $this->limit);
        }

        $ids = $this->getIdsWithinDelta($all);
        $total = \count($ids);

        (new EachPromise($this->chunkedDetailPromiseGenerator($ids), [
            'concurrency' => $this->parameters['concurrency'] ?? 10,
            'fulfilled' => function (array $data) use ($block, $total) {
                foreach ($data['tour'] ?? [] as $tour) {
                    $block($this->tourDataToAdapter($tour), $total);
                }
            }
        ]))->promise()->wait();

        $result = new ServiceResult();
        $result->setAll($this->listItemsToIds($all));
        return $result;
    }

    private function fetchTourList(): ?array
    {
        $uri = $this->prepareUri(static::PATH_TOURS_LIST, []);
        $data = $this->jsonRequest($uri, 5)->wait();
        return $data['data'] ?? null;
    }

    private function getIdsWithinDelta(array $tourList): array
    {
        $now = (new \DateTime())->getTimestamp();
        $within = array_filter(
            $tourList,
            function (array $item) use ($now) {
                $date = new \DateTime($item['lastModified']);
                return $date->add($this->delta)->getTimestamp() > $now;
            }
        );

        return array_map(
            function (array $item) {
                return $item['id'];
            },
            $within
        );
    }

    private function chunkedDetailPromiseGenerator(array $ids): \Generator
    {
        $chunks = array_chunk($ids, static::RECORDS_PER_DETAIL_REQUEST);
        foreach ($chunks as $chunk) {
            yield $this->jsonRequest(
                $this->prepareUri(static::PATH_TOUR_DETAIL, [ ':ids' => implode(',', $chunk) ]),
                5
            );
        }
    }

    /** @return string[] */
    private function listItemsToIds(array $listItems): array
    {
        return array_map(
            function (array $item) {
                return (new TourAdapter($item))->getExternalId();
            },
            $listItems
        );
    }

    protected function prepareUri(string $uri, array $parameters): Uri
    {
        $parameters = array_merge(
            $parameters,
            [
                ':apiKey' => $this->apiKey,
                ':clientName' => $this->clientName,
                ':language' => $this->language
            ]
        );

        $uri = str_replace(array_keys($parameters), array_values($parameters), $uri);
        return new Uri($uri);
    }

    private function tourDataToAdapter(array $tour): TourAdapter
    {
        $adapter = new TourAdapter($tour);
        if ($this->language !== null) {
            $adapter->setLanguage($this->language);
        }
        return $adapter;
    }
}
