<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Service\Toubiz\Legacy\TourApiService;

use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\MonthConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\TourApiService\TourAdapter;
use PHPUnit\Framework\TestCase;
use Psr\Log\Test\TestLogger;

class TourAdapterTest extends TestCase
{

    /** @dataProvider provideValidGpxFiles */
    public function testGetsGeometryFromGpxFile(string $file): void
    {
        [ $tour ] = $this->adapterForGpsFile($file);

        $attributes = $tour->getAttributes();
        $attributeNames = array_map(function(AttributeAdapterInterface $attribute) {
            return $attribute->getName();
        }, $attributes);
        $this->assertContains(TourAttributes::GEOMETRY, $attributeNames, 'Attributes should include geometry');
    }

    /** @dataProvider provideErroneousGpxFiles */
    public function testErrorWhenParsingGpxFileDoesNotCompletelyAbortExecution(string $file): void
    {
        [ $tour, $logger ] = $this->adapterForGpsFile($file);
        $attributes = $tour->getAttributes();
        $this->assertArrayNotHasKey(TourAttributes::GEOMETRY, $attributes);

        $messages = array_map(function(array $record) {
            return $record['message'];
        }, $logger->records);
        $this->assertContains('Error parsing GPS File', implode('|', $messages));
    }

    /** @dataProvider provideInvalidGpxFiles */
    public function testInvalidOrEmptyGpxFilesDoNotAddGeometry(string $file): void
    {
        [ $tour ] = $this->adapterForGpsFile($file);

        $attributes = $tour->getAttributes();
        $this->assertArrayNotHasKey(TourAttributes::GEOMETRY, $attributes);
    }

    public function provideErroneousGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/erroneous-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return $data;
    }

    public function provideValidGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/valid-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return $data;
    }

    public function provideInvalidGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/invalid-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return array_merge($data, $this->provideErroneousGpxFiles());
    }

    private function adapterForGpsFile(string $file): array
    {
        $tour = new TourAdapter([
            'remote_id' => 'TEST__' . preg_replace('/\W/', '-', $file),
            'object' => [
                'data_map' => [
                    'geodata_file_gpx' => [
                        'content' => [
                            'path' => $file
                        ]
                    ]
                ]
            ]
        ]);

        $logger = new TestLogger();
        $tour->setLogger($logger);

        return [ $tour, $logger ];
    }

    /** @dataProvider provideDescriptions */
    public function testProperlyStripDescription(string $expected, string $input): void
    {
        $adapter = new TourAdapter(
            [
                'object' => [
                    'data_map' => [
                        'description' => [
                            'content' => $input
                        ]
                    ]
                ]
            ]
        );
        $this->assertEquals($expected, $adapter->getDescription());
    }

    public function testInterpretsFullYearCorrectly(): void
    {
        $adapter = new TourAdapter([
            'remote_id' => 'test',
            'object' => [
                'data_map' => [
                    'season' => [
                        'datatypestring' => 'ezselection',
                        'content' => 'ganzjährig',
                        'name' => 'Empfohlene Jahreszeit'
                    ]
                ]
            ]
        ]);

        $bestSeasonAttributes = array_filter($adapter->getAttributes(), function(AttributeAdapterInterface $attribute) {
            return $attribute->getName() === TourAttributes::BEST_SEASON;
        });
        $bestSeasons = array_map(function(AttributeAdapterInterface $attribute) {
            return $attribute->getData();
        }, $bestSeasonAttributes);

        $this->assertCount(12, $bestSeasonAttributes);
        $this->assertContains(MonthConstants::JANUARY, $bestSeasons);
        $this->assertContains(MonthConstants::FEBRUARY, $bestSeasons);
        $this->assertContains(MonthConstants::MARCH, $bestSeasons);
        $this->assertContains(MonthConstants::APRIL, $bestSeasons);
        $this->assertContains(MonthConstants::MAY, $bestSeasons);
        $this->assertContains(MonthConstants::JUNE, $bestSeasons);
        $this->assertContains(MonthConstants::JULY, $bestSeasons);
        $this->assertContains(MonthConstants::AUGUST, $bestSeasons);
        $this->assertContains(MonthConstants::SEPTEMBER, $bestSeasons);
        $this->assertContains(MonthConstants::OCTOBER, $bestSeasons);
        $this->assertContains(MonthConstants::NOVEMBER, $bestSeasons);
        $this->assertContains(MonthConstants::DECEMBER, $bestSeasons);
    }

    public function provideDescriptions(): array
    {
        return [
            'valid text remains text' => ['text', 'text'],
            'invalid script is completely discarded' => ['', '<script>text</script>'],
            'invalid tag is removed' => ['text', '<div>text</div>'],
            'invalid custom tag is removed' => ['text', '<myTag>text</myTag>'],
            'invalid empty tag' => ['', '<u></u>'],
            'valid code stays unchanged' => ['<u>text</u>', '<u>text</u>'],
            'invalid post spacing is removed' => ['<u>text</u>', '<u >text</u>'],
            'invalid data attribute' => ['<u>text</u>', '<u data>text</u>'],
            'invalid data attribute with value' => ['<u>text</u>', '<u data="foo">text</u>'],
            'invalid attribute for tag (valid uri)' =>
                ['<u>text</u>', '<u href="www.example.com">text</u>'],
            'invalid data attribute followed by invalid attribute for tag' =>
                ['<u>text</u>', '<u href="www.example.com" data>text</u>'],
            'invalid attribute for tag followed by invalid data attribute' =>
                ['<u>text</u>', '<u data href="www.example.com">text</u>'],
            'invalid attribute for tag (title)' =>
                ['<u>text</u>', '<u title="Title">text</u>'],
            'invalid attribute for tag (target)' =>
                ['<u>text</u>', '<u target="_blank">text</u>'],
            'invalid a tag without attribute href' =>
                ['text', '<a>text</a>'],
            'valid a tag with href' =>
                ['<a href="www.example.com">text</a>', '<a href="www.example.com">text</a>'],
            'invalid a tag using single quotes' =>
                ['<a href="www.example.com">text</a>', '<a href=\'www.example.com\'>text</a>'],
            'invalid duplicate href' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" href="www.example-2.com">text</a>'],
            'invalid duplicate href with invalid data attribute in between' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" data href="www.example-2.com">text</a>'],
            'valid a tag with title' =>
                ['<a href="www.example.com" title="Title">text</a>',
                    '<a href="www.example.com" title="Title">text</a>'],
            'valid a tag with target' =>
                ['<a href="www.example.com" target="_blank" rel="noreferrer noopener">text</a>',
                    '<a href="www.example.com" target="_blank">text</a>'],
            'valid nesting is kept' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</u></strong>'],
            'invalid nesting (wrong order) is corrected' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</strong></u>'],
            'invalid nesting (missing closing tags) are added in the end' =>
                ['<strong><u><em>text</em></u></strong>', '<strong><u><em>text</em></strong>'],
            'invalid nesting (only closing tag) is removed' =>
                ['<strong><em>text</em></strong>', '<strong><em>text</em></u></strong>'],
        ];
    }



    public function testIgnoresMediaWithEmptySourceUri(): void
    {
        $adapter = new TourAdapter([
              'remote_id' => 'test',
              'object' => [
                  'data_map' => [
                      'gallery' => [
                          'content' => [
                              [ 'id' => 'test', 'reference' => '' ],
                              [ 'id' => 'test', 'reference' => 'https://placehold.it/500x500' ]
                          ]
                      ]
                  ]
              ]
          ]);

        $this->assertCount(1, $adapter->getMedia());
        $this->assertEquals('https://placehold.it/500x500', $adapter->getMedia()[0]->getSourceUri());
    }

    public function testProvidesRemoteIdsAsCityRelations(): void
    {
        $adapter = new TourAdapter([
           'remote_id' => 'test',
           'object' => [
               'data_map' => [
                   'city_relation' => [
                       'datatypestring' => 'ezobjectrelationlist',
                       'name' => 'Zugehörige Orte',
                       'content' => 'City 1#City 2#City 3',
                       'remote_ids' => 'location_99#location_42#location_2'
                   ]
               ]
           ]
       ]);

        $this->assertCount(3, $adapter->getCitySelectors());
        $this->assertEquals(ExternalIdType::TOUBIZ_LEGACY, $adapter->getCitySelectors()[0]->getType());
        $this->assertEquals('99', $adapter->getCitySelectors()[0]->getId());
        $this->assertEquals(ExternalIdType::TOUBIZ_LEGACY, $adapter->getCitySelectors()[1]->getType());
        $this->assertEquals('42', $adapter->getCitySelectors()[1]->getId());
        $this->assertEquals(ExternalIdType::TOUBIZ_LEGACY, $adapter->getCitySelectors()[2]->getType());
        $this->assertEquals('2', $adapter->getCitySelectors()[2]->getId());
    }


    public function testMapsSearchString(): void
    {
        $adapter = new TourAdapter([
             'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
             'object' => [
                'data_map' => [
                    'searchwords' => [
                        'datatypestring' => 'ezkeyword',
                        'content' => 'foo, bar, baz',
                    ]
                ]
            ]
         ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayHasKey('additionalSearchString', $attributes);
        $this->assertEquals('foo, bar, baz', $attributes['additionalSearchString']);
    }

    public function testDoesNotMapSearchStringsWhenNonExistent(): void
    {
        $adapter = new TourAdapter([
           'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
           'object' => [ 'data_map' => [] ]
       ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayNotHasKey('additionalSearchString', $attributes);
    }

    public function testDoesNotMapSearchStringsWhenEmpty(): void
    {
        $adapter = new TourAdapter([
           'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
           'object' => [
               'data_map' => [
                   'searchwords' => [
                       'datatypestring' => 'ezkeyword',
                       'content' => '',
                   ]
               ]
           ]
       ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayNotHasKey('additionalSearchString', $attributes);
    }


    public function testDoesNotMapSearchStringsWhenOnlyWhitespace(): void
    {
        $adapter = new TourAdapter([
           'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
           'object' => [
               'data_map' => [
                   'searchwords' => [
                       'datatypestring' => 'ezkeyword',
                       'content' => '   ',
                   ]
               ]
           ]
       ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayNotHasKey('additionalSearchString', $attributes);
    }
}
