<?php declare(strict_types=1);
namespace Newland\ToubizApi\Tests\Unit\Service\Toubiz\Legacy\ObjectAdapter\CityApiService;


use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\CityAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\CityApiService\CityAdapter;
use PHPUnit\Framework\TestCase;

class CityAdapterTest extends TestCase
{

    /**
     * @dataProvider citiesApiResponseDataProvider
     * @param array $data
     */
    public function testHasNoBookingUri(array $data): void
    {
        $adapter = new CityAdapter($data);
        $this->assertNull($adapter->getBookingUri());
    }

    /**
     * @dataProvider citiesApiResponseDataProvider
     * @param array $data
     */
    public function testHasNoDetailUri(array $data): void
    {
        $adapter = new CityAdapter($data);
        $this->assertNull($adapter->getDetailUri());
    }

    /**
     * @dataProvider citiesApiResponseDataProvider
     * @param array $data
     */
    public function testMapsMediaIntoAdapters(array $data): void
    {
        $adapter = new CityAdapter($data);
        $this->assertGreaterThan(0, $adapter->getMedia());
        foreach ($adapter->getMedia() as $medium) {
            $this->assertInstanceOf(MediumAdapterInterface::class, $medium);
        }
    }

    /**
     * @dataProvider citiesApiResponseDataProvider
     * @param array $data
     */
    public function testUsesConstantsAsAttributeNames(array $data): void
    {
        $constantValues = (new \ReflectionClass(CityAttributes::class))->getConstants();

        $adapter = new CityAdapter($data);
        $this->assertGreaterThan(0, $adapter->getAttributes());
        foreach ($adapter->getAttributes() as $attribute) {
            $this->assertInstanceOf(AttributeAdapterInterface::class, $attribute);
            $this->assertContains($attribute->getName(), $constantValues, 'Attribute names must be in CityAttributes');
        }
    }


    public function citiesApiResponseDataProvider(): array
    {
        $json = json_decode(file_get_contents(__DIR__ . '/../../../../../../Fixtures/ToubizLegacyData/cityData.json'), true);
        return array_map(
            function(array $child) { return [ $child ]; },
            $json['content']['children_list']
        );
    }

    public function provideInvalidAndValidUrls(): array
    {
        return [
            [false, 'HaTeTePeEs://www.example.com'],
            [false, 'http//www.example.com'],
            [false, '://www.example.com'],
            [false, 'example.com disjunct'],

            [true, 'https://example.com'],
            [true, 'http://www.example.com'],
            [true, 'www.example.com/file.php?value=1'],
            [true, 'example.com#main'],
        ];
    }

    public function provideCitiesApiResponsesAndUrls(): array
    {
        $cities = $this->citiesApiResponseDataProvider();
        $urls = $this->provideInvalidAndValidUrls();
        $combined = [];
        foreach ($cities as $city) {
            foreach ($urls as $url) {
                $combined[] = [ $city[0], $url[0], $url[1] ];
            }
        }
        return $combined;
    }

    /**
     * @dataProvider provideCitiesApiResponsesAndUrls
     * @param array $data
     * @param bool $valid
     * @param string $url
     */
    public function testDismissesInvalidUrlsForSocialLinks(array $data, bool $valid, string $url): void
    {
        $data['object']['data_map']['flickr'] = [ 'content' => $url ];
        $adapter = new CityAdapter($data);

        if ($valid) {
            $this->assertEquals($url, $adapter->getFlickrUri());
        } else {
            $this->assertNull($adapter->getFlickrUri());
        }
    }

    /** @dataProvider provideDescriptions */
    public function testProperlyStripDescription(string $expected, string $input): void
    {
        $adapter = new CityAdapter(
            [
                'object' => [
                    'data_map' => [
                        'description' => [
                            'content' => $input
                        ]
                    ]
                ]
            ]
        );
        $this->assertEquals($expected, $adapter->getDescription());
    }

    public function provideDescriptions(): array
    {
        return [
            'valid text remains text' => ['text', 'text'],
            'invalid script is completely discarded' => ['', '<script>text</script>'],
            'invalid tag is removed' => ['text', '<div>text</div>'],
            'invalid custom tag is removed' => ['text', '<myTag>text</myTag>'],
            'invalid empty tag' => ['', '<u></u>'],
            'valid code stays unchanged' => ['<u>text</u>', '<u>text</u>'],
            'invalid post spacing is removed' => ['<u>text</u>', '<u >text</u>'],
            'invalid data attribute' => ['<u>text</u>', '<u data>text</u>'],
            'invalid data attribute with value' => ['<u>text</u>', '<u data="foo">text</u>'],
            'invalid attribute for tag (valid uri)' =>
                ['<u>text</u>', '<u href="www.example.com">text</u>'],
            'invalid data attribute followed by invalid attribute for tag' =>
                ['<u>text</u>', '<u href="www.example.com" data>text</u>'],
            'invalid attribute for tag followed by invalid data attribute' =>
                ['<u>text</u>', '<u data href="www.example.com">text</u>'],
            'invalid attribute for tag (title)' =>
                ['<u>text</u>', '<u title="Title">text</u>'],
            'invalid attribute for tag (target)' =>
                ['<u>text</u>', '<u target="_blank">text</u>'],
            'invalid a tag without attribute href' =>
                ['text', '<a>text</a>'],
            'valid a tag with href' =>
                ['<a href="www.example.com">text</a>', '<a href="www.example.com">text</a>'],
            'invalid a tag using single quotes' =>
                ['<a href="www.example.com">text</a>', '<a href=\'www.example.com\'>text</a>'],
            'invalid duplicate href' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" href="www.example-2.com">text</a>'],
            'invalid duplicate href with invalid data attribute in between' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" data href="www.example-2.com">text</a>'],
            'valid a tag with title' =>
                ['<a href="www.example.com" title="Title">text</a>',
                    '<a href="www.example.com" title="Title">text</a>'],
            'valid a tag with target' =>
                ['<a href="www.example.com" target="_blank" rel="noreferrer noopener">text</a>',
                    '<a href="www.example.com" target="_blank">text</a>'],
            'valid nesting is kept' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</u></strong>'],
            'invalid nesting (wrong order) is corrected' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</strong></u>'],
            'invalid nesting (missing closing tags) are added in the end' =>
                ['<strong><u><em>text</em></u></strong>', '<strong><u><em>text</em></strong>'],
            'invalid nesting (only closing tag) is removed' =>
                ['<strong><em>text</em></strong>', '<strong><em>text</em></u></strong>'],
        ];
    }



    public function testIgnoresMediaWithEmptySourceUri(): void
    {
        $adapter = $this->adapterWithDataMap([
            'gallery' => [
                [ 'id' => 'test', 'reference' => '' ],
                [ 'id' => 'test', 'reference' => 'https://placehold.it/500x500' ]
            ]
        ]);

        $this->assertCount(1, $adapter->getMedia());
        $this->assertEquals('https://placehold.it/500x500', $adapter->getMedia()[0]->getSourceUri());
    }

    public function testMapsTomasId(): void
    {
        $adapter = $this->adapterWithDataMap([ 'tomas_id' => 'foobar' ]);
        $this->assertCount(1, $adapter->getAdditionalExternalIds());
        $this->assertEquals(ExternalIdType::TOMAS, $adapter->getAdditionalExternalIds()[0]->getType());
        $this->assertEquals('foobar', $adapter->getAdditionalExternalIds()[0]->getId());
    }

    public function testMapsOutdoorActiveId(): void
    {
        $adapter = $this->adapterWithDataMap([ 'alpstein_id' => 'foobar' ]);
        $this->assertCount(1, $adapter->getAdditionalExternalIds());
        $this->assertEquals(ExternalIdType::OUTDOORACTIVE, $adapter->getAdditionalExternalIds()[0]->getType());
        $this->assertEquals('foobar', $adapter->getAdditionalExternalIds()[0]->getId());
    }

    private function adapterWithDataMap(array $datamap): CityAdapter
    {
        $data = [
            'remote_id' => 'test',
            'object' => [ 'data_map' => [] ]
        ];

        foreach ($datamap as $key => $value) {
            $data['object']['data_map'][$key] = [ 'content' => $value, ];
        }

        return new CityAdapter($data);
    }


    public function testMapsSearchString(): void
    {
        $adapter = new CityAdapter([
           'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
           'object' => [
               'data_map' => [
                   'searchwords' => [
                       'datatypestring' => 'ezkeyword',
                       'content' => 'foo, bar, baz',
                   ]
               ]
           ]
       ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayHasKey('additionalSearchString', $attributes);
        $this->assertEquals('foo, bar, baz', $attributes['additionalSearchString']);
    }

    public function testDoesNotMapSearchStringsWhenNonExistent(): void
    {
        $adapter = new CityAdapter([
           'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
           'object' => [ 'data_map' => [] ]
       ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayNotHasKey('additionalSearchString', $attributes);
    }

    public function testDoesNotMapSearchStringsWhenEmpty(): void
    {
        $adapter = new CityAdapter([
           'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
           'object' => [
               'data_map' => [
                   'searchwords' => [
                       'datatypestring' => 'ezkeyword',
                       'content' => '',
                   ]
               ]
           ]
       ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayNotHasKey('additionalSearchString', $attributes);
    }


    public function testDoesNotMapSearchStringsWhenOnlyWhitespace(): void
    {
        $adapter = new CityAdapter([
           'remote_id' => 'b93d29f288307f7f59c8ddcad09cf1a3',
           'object' => [
               'data_map' => [
                   'searchwords' => [
                       'datatypestring' => 'ezkeyword',
                       'content' => '   ',
                   ]
               ]
           ]
       ]);

        $attributes = [];
        foreach ($adapter->getAttributes() as $attribute) {
            $attributes[$attribute->getName()] = $attribute->getData();
        }

        $this->assertArrayNotHasKey('additionalSearchString', $attributes);
    }
}
