<?php
namespace Newland\Toubiz\Api\Service\OpenWeatherMap\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\Weather\DayAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Weather\HourAdapterInterface;
use Newland\Toubiz\Api\Service\Weather\ObjectAdapter\AbstractDayAdapter;

class DayAdapter extends AbstractDayAdapter implements DayAdapterInterface
{
    /**
     * @var HourAdapter[]
     */
    protected $hours = [];
    /**
     * @var \DateTime
     */
    private $date;
    /**
     * @var float
     */
    private $maximumTemperature;
    /**
     * @var float
     */
    private $minimumTemperature;
    /**
     * @var string
     */
    private $symbol;
    /**
     * @var int
     */
    private $sunshineDuration;

    public function __construct(array $data)
    {
        $this->parseData($data);
    }

    private function parseData(array $data)
    {
        $date = \DateTime::createFromFormat('U', array_slice($data, 0, 1)[0]['dt']);
        if ($date) {
            $this->date = $date;
        }

        foreach ($data as $hourItem) {
            $this->hours[] = new HourAdapter($hourItem);
        }

        $this->symbol = $this->getGeneralSymbol();
        $this->sunshineDuration = $this->calculateSunshineDuration($data);

        $maximumTemperature = $this->calculateMaximumTemperature();
        if ($maximumTemperature) {
            $this->maximumTemperature = $maximumTemperature;
        }

        $minimumTemperature = $this->calculateMinimumTemperature();
        if ($minimumTemperature) {
            $this->minimumTemperature = $minimumTemperature;
        }
    }

    protected function calculateSunshineDuration(array $data): int
    {
        return 0;
    }

    /**
     * Returns the external id used by the API
     * to identify the record.
     *
     * @return string
     */
    public function getExternalId(): string
    {
        return '';
    }

    public function getDate(): \DateTime
    {
        return $this->date;
    }

    /**
     * @return HourAdapterInterface[]
     */
    public function getAllHours(): array
    {
        return $this->hours;
    }

    /**
     * Key hours spread over the course of a day,
     * usually 00:00, 06:00, 12:00 and 18:00.
     *
     * @return HourAdapterInterface[]
     */
    public function getMainHours(): array
    {
        $mainHours = [];

        foreach ($this->hours as $hourItem) {
            if (in_array($hourItem->getTime(), [6, 9, 12, 15, 18, 21])) {
                $mainHours[] = $hourItem;
            }
        }

        return $mainHours;
    }

    public function getHoursForCalculation(): array
    {
        return $this->getMainHours();
    }

    public function getMainHoursPadded(): array
    {
        $mainHours = $this->getMainHours();

        while (\count($mainHours) < 6) {
            $mainHours = array_merge([null], $mainHours);
        }

        return $mainHours;
    }

    public function getMaximumTemperature(): float
    {
        return $this->maximumTemperature;
    }

    public function getMinimumTemperature(): float
    {
        return $this->minimumTemperature;
    }

    public function getSymbol(): string
    {
        return $this->symbol;
    }

    /**
     * The total amount of sunshine during the day in minutes.
     *
     * @return int
     */
    public function getSunshineDuration(): int
    {
        return $this->sunshineDuration;
    }

    public function hasSunshineDurationData(): bool
    {
        return false;
    }
}
