<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Exception\InvalidJsonException;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DirectMarketerApiService\DirectMarketerAdapter;

/**
 * Service for legacy Toubiz gastronomy API.
 *
 * Concrete implementation to communicate with
 * the first, "old" toubiz API returning gastronomy articles.
 */
class DirectMarketerApiService extends AbstractService
{
    use LanguageAware, Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://api.toubiz.de/json/direkt/';

    public function fetchDirectMarketers(callable $block): ServiceResult
    {
        $result = new ServiceResult();
        $concurrency = $this->parameters['concurrency'] ?? 10;
        $perPage = $this->parameters['perPage'] ?? 100;

        $pool = new ConcurrentPaginatedRequests(
            $concurrency,
            function (int $page) use ($perPage, $block, $result) {
                if (!$this->withinLimit($page * $perPage)) {
                    return new RejectedPromise('Page is outside of limit');
                }
                return $this->sendRequest(($page - 1) * $perPage, $perPage)->then(
                    function (array $data) use ($block, $result) {
                        $result->addDeleted($data['deleted'] ?? []);
                        foreach ($data['result']['item'] as $item) {
                            $block($this->directeMarketerDataToAdapter($item), $data['itemCount'] ?? null);
                        }
                    }
                );
            }
        );

        $pool->start()->wait();
        return $result;
    }

    protected function sendRequest(int $offset, int $limit): PromiseInterface
    {
        $url = Uri::withQueryValues(
            new Uri('modified.php'),
            [
                'key' => $this->apiKey,
                'modified' => $this->delta->asSeconds(),
                'offset' => $offset,
                'limit' => $limit,
            ]
        );

        return $this->jsonRequest(
            $url,
            5,
            function (array $data) {
                return !empty($data['result']);
            }
        );
    }

    private function directeMarketerDataToAdapter(array $item): DirectMarketerAdapter
    {
        $article = new DirectMarketerAdapter($item);
        if ($this->language) {
            $article->setLanguage($this->language);
        }
        return $article;
    }
}
