<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\TourApiService;

use Newland\GpsFileParsing\Helper\XmlFileReader;
use Newland\GpsFileParsing\Model\Point;
use Newland\GpsFileParsing\ParserFactory;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\MonthConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\LegacyToubizArticleAdapterCommon;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\UsesCategoryMap;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;
use Psr\Log\LoggerAwareTrait;

class TourAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface
{
    use LanguageAware,
        LoggerAwareTrait,
        UsesFirstMediumAsMainMedium,
        LegacyToubizArticleAdapterCommon,
        UsesCategoryMap;

/**
 * Necessary mapping for Toubiz tours to have the same entries as OutdoorActive
 *
 */

    protected $tourTypeMapping = [
        'Rundtour' => 'loopTour',
        'Streckentour' => 'oneWayTour',
    ];
    protected $dataMapMapping = [
        TourAttributes::STARTING_POINT => 'startingpoint',
        TourAttributes::ENDING_POINT => 'endpoint',
        TourAttributes::TECHNIQUE_RATING => 'severity',
        TourAttributes::LANDSCAPE_RATING => 'beauty',
        TourAttributes::DIFFICULTY_RATING => 'degree_of_difficulty',
        TourAttributes::CONDITION_RATING => 'fitness',
        TourAttributes::MAXIMUM_ALTITUDE => 'highestpoint',
        TourAttributes::ASCEND_ELEVATION => 'uphill',
        TourAttributes::DESCENT_ELEVATION => 'downhill',
        TourAttributes::EXPERIENCE_RATING => 'popularity',
        TourAttributes::SAFETY_NOTE => 'special',
        TourAttributes::BEST_SEASON => 'season',
        TourAttributes::MINIMUM_ALTITUDE => 'lowest_point',


        // Note: Not all attributes are in toubiz tours available!
        TourAttributes::ISCLOSED => null,
        TourAttributes::PRIVATE_TRANSIT => null,
        TourAttributes::PUBLIC_TRANSIT => null,
        TourAttributes::NECESARRY_EQUIPMENT => null,
        TourAttributes::DIRECTIONS => null,
        TourAttributes::PARKING => null,
        TourAttributes::ADDITIONAL_INFORMATION => null,
        TourAttributes::HINT_OF_AUTHOR => null,
    ];

    protected $monthMapping = [
        'Januar' => MonthConstants::JANUARY,
        'Februar' => MonthConstants::FEBRUARY,
        'März' => MonthConstants::MARCH,
        'Maerz' => MonthConstants::MARCH,
        'Marz' => MonthConstants::MARCH,
        'April' => MonthConstants::APRIL,
        'Mai' => MonthConstants::MAY,
        'Juni' => MonthConstants::JUNE,
        'Juli' => MonthConstants::JULY,
        'August' => MonthConstants::AUGUST,
        'September' => MonthConstants::SEPTEMBER,
        'Oktober' => MonthConstants::OCTOBER,
        'November' => MonthConstants::NOVEMBER,
        'Dezember' => MonthConstants::DECEMBER,
    ];

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_TOUR;
    }

    protected function parseAttributes(): array
    {
        $attributes = $this->mapDataMapEntries($this->dataMapMapping);

        if ($this->object['object']['data_map']['season']['content'] ?? false) {
            if (\in_array('ganzjährig', $attributes[TourAttributes::BEST_SEASON], true)) {
                $attributes[TourAttributes::BEST_SEASON] = MonthConstants::ALL;
            } else {
                $attributes[TourAttributes::BEST_SEASON] = array_filter(array_map(
                    function ($month) {
                        return $this->monthMapping[$month] ?? null;
                    },
                    $attributes[TourAttributes::BEST_SEASON] ?? []
                ));
            }
        }

        $attributes[TourAttributes::DATA_SOURCE] = TourAttributes::DATA_SOURCE_TOUBIZ_LEGACY;

        $geometry = $this->extractGeometry($this->object['object']['data_map']);
        if (!empty($geometry)) {
            $attributes[TourAttributes::GEOMETRY] = $geometry;
        }

        $length = $this->object['object']['data_map']['length']['content'] ?? null;
        if ($length) {
            $attributes[TourAttributes::TOUR_LENGTH] = $length * 1000;
        }
        $duration = $this->object['object']['data_map']['duration']['content'] ?? null;
        if ($duration) {
            $attributes[TourAttributes::TOUR_DURATION] = $duration * 60;
        }
        $tourType = $this->object['object']['data_map']['tour_typ']['content'] ?? null;
        if ($tourType) {
            if (array_key_exists($tourType, $this->tourTypeMapping)) {
                $attributes['properties'] = $this->tourTypeMapping[$tourType];
            } else {
                $attributes['properties'] = $tourType;
            }
        }

        return $attributes;
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->extractMediaFromGallery(function (array $data) {
            return new MediumAdapter($data);
        });
    }


    /**
     * @return AttributeAdapter[]
     */
    public function getAttributes(): array
    {
        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    public function getFiles(): array
    {
        return [];
    }

    /**
     * Returns the URI for an external booking page.
     *
     * @return string|null
     */
    public function getBookingUri(): ?string
    {
        return null;
    }

    /**
     * Returns the URI for an external detail page.
     *
     * @return string|null
     */
    public function getDetailUri(): ?string
    {
        return null;
    }

    /**
     * The complete, encoded opening times data.
     *
     * @return string|null
     */
    public function getOpeningTimes(): ?string
    {
        return null;
    }

    /**
     * The format of the opening times string.
     *
     * @see \Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants for values.
     *
     * @return string|null
     */
    public function getOpeningTimesFormat(): ?string
    {
        return null;
    }

    /**
     * Return the average rating on a scale of 0-100
     *
     * @return int|null
     */
    public function getAverageRating(): ?int
    {
        return null;
    }

    /**
     * Return the number of ratings.
     *
     * @return int|null
     */
    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    /**
     * @return CategoryAdapterInterface[]
     */
    public function getCategories(): array
    {
        return $this->extractCategories(function (array $properties) {
            return new CategoryAdapter($properties);
        });
    }

    public function getDescription(): ?string
    {
        $value = $this->object['object']['data_map']['description']['content'] ?? null;
        return $value ? $this->stringCleaner->purifyHtml((string) $value) : null;
    }

    public function extractGeometry(array $data): array
    {
        $geoFile = $data['geodata_file_gpx']['content'] ??
            $data['geodata_file']['content'] ?? null;

        if (($geoFile['path'] ?? null) === null) {
            return [];
        }

        try {
            $track = (new ParserFactory(new XmlFileReader()))
                ->resolveParser($geoFile['path'])
                ->extractTrack($geoFile['path']);

            return $track->mapPoints(function (Point $point) {
                return [ $point->getLatitude(), $point->getLongitude() ];
            });
        } catch (\Exception $e) {
            $this->logger->error(sprintf(
                'Tour [%s]: Error parsing GPS File "%s": [%s] %s. Track data from this file will be ignored.',
                $this->object['remote_id'] ?? '???',
                $geoFile['path'],
                \get_class($e),
                $e->getMessage()
            ));
            return [];
        }
    }

    public function getCitySelectors(): array
    {
        $ids = explode('#', $this->object['object']['data_map']['city_relation']['remote_ids'] ?? '');
        $selectors = [];

        foreach ($ids as $id) {
            $id = str_replace('location_', '', $id);
            if (!empty($id)) {
                $selectors[] = new ExternalIdSelector(ExternalIdType::TOUBIZ_LEGACY, $id);
            }
        }

        return $selectors;
    }
}
