<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\MonthConstants;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\FileAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\WithCacheProperty;

/**
 * Tour adapter.
 *
 * Describes a tour.
 */
class TourAdapter extends AbstractObjectAdapter implements ArticleAdapterInterface, ArticleWithLocationDataInterface
{
    use LanguageAware;
    use WithCacheProperty;

    /** @var array|null */
    private $cachedGeometry;

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE . '_' . $this->object['id'];
    }

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_TOUR;
    }

    public function getName(): string
    {
        $name = $this->object['title'] ?? null;
        return strip_tags((string) $name);
    }

    public function getAbstract(): ?string
    {
        $value = $this->object['shortText'] ?? null;
        return $value ? $this->stringCleaner->purifyHtml((string) $value) : null;
    }

    public function getDescription(): ?string
    {
        $value = $this->object['longText'] ?? null;
        return $value ? $this->stringCleaner->purifyHtml((string) $value) : null;
    }

    public function getLatitude(): ?float
    {
        $geometry = $this->extractGeometry();
        if (\count($geometry) > 0) {
            return $geometry[0][0];
        }

        return null;
    }

    public function getLongitude(): ?float
    {
        $geometry = $this->extractGeometry();
        if (\count($geometry) > 0) {
            return $geometry[0][1];
        }

        return null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        $geometry = $this->extractGeometry();
        $object = $this->object;
        if (\count($geometry) > 0) {
            $object['firstGeometry'] = [
                'lat' => $geometry[0][0],
                'lon' => $geometry[0][1],
            ];
        }
        return new AddressAdapter($object);
    }

    /**
     * @return CategoryAdapterInterface[]
     */
    public function getCategories(): array
    {
        $category = $this->object['category'] ?? null;
        return $category ? [ new CategoryAdapter($category) ] : [];
    }

    /**
     * @return MediumAdapterInterface[]
     */
    public function getMedia(): array
    {
        return $this->cache('media', function () {
            return array_map(
                function ($data) {
                    return new MediumAdapter($data);
                },
                $this->object['images']['image'] ?? []
            );
        });
    }

    public function getMainMedium(): ?MediumAdapterInterface
    {
        $images = $this->getMedia();

        if (empty($images)) {
            return null;
        }

        // Set the first image as the default main image.
        $primaryImage = $images[0];
        $externalId = $this->object['primaryImage']['id'] ?? null;
        if ($externalId === null) {
            return $primaryImage;
        }

        $primaryImageId = TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE . '_' . $externalId;
        foreach ($images as $image) {
            if ($image->getExternalId() === $primaryImageId) {
                $primaryImage = $image;
                break;
            }
        }

        return $primaryImage;
    }

    /**
     * @return FileAdapterInterface[]
     */
    public function getFiles(): array
    {
        return [];
    }

    /**
     * @return bool
     */
    public function hasAttributes(): bool
    {
        return true;
    }

    /**
     * @return array
     */
    public function getAttributes(): array
    {
        $items = [
            new AttributeAdapter($this->getExternalId(), 'geometry', $this->extractGeometry()),
        ];

        foreach ($this->object['properties']['property'] ?? [] as $property) {
            $items[] = new AttributeAdapter(
                $this->getExternalId(),
                'properties',
                (string) $property['tag']
            );
        }

        foreach ($this->object['labels'] ?? [] as $attribute => $value) {
            $items[] = new AttributeAdapter($this->getExternalId(), 'labels', $attribute);
        }


        foreach ($this->object['crossCountryTechnique']['property'] ?? [] as $technique) {
            $items[] = new AttributeAdapter(
                $this->getExternalId(),
                'crossCountryTechnique',
                (string) $technique['tag']
            );
        }


        $months = [
            'jan' => MonthConstants::JANUARY,
            'feb' => MonthConstants::FEBRUARY,
            'mar' => MonthConstants::MARCH,
            'apr' => MonthConstants::APRIL,
            'may' => MonthConstants::MAY,
            'jun' => MonthConstants::JUNE,
            'jul' => MonthConstants::JULY,
            'aug' => MonthConstants::AUGUST,
            'sep' => MonthConstants::SEPTEMBER,
            'oct' => MonthConstants::OCTOBER,
            'nov' => MonthConstants::NOVEMBER,
            'dec' => MonthConstants::DECEMBER,
        ];

        foreach ($months as $external => $internal) {
            if ($this->object['season'][$external] ?? false) {
                $items[] = new AttributeAdapter($this->getExternalId(), 'bestSeason', $internal);
            }
        }

        $simpleAttributes = [
            TourAttributes::ASCEND_ELEVATION => (int) ($this->object['elevation']['ascent'] ?? 0),
            TourAttributes::DESCENT_ELEVATION => (int) ($this->object['elevation']['descent'] ?? 0),
            TourAttributes::CONDITION_RATING => (int) ($this->object['rating']['condition'] ?? 0),
            TourAttributes::DIFFICULTY_RATING => (int) ($this->object['rating']['difficulty'] ?? 0),
            TourAttributes::NECESARRY_EQUIPMENT =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['equipment'] ?? '')),
            TourAttributes::EXPERIENCE_RATING => (int) ($this->object['rating']['qualityOfExperience'] ?? 0),
            TourAttributes::LANDSCAPE_RATING => (int) ($this->object['rating']['landscape'] ?? 0),
            TourAttributes::MAXIMUM_ALTITUDE => (int) ($this->object['elevation']['maxAltitude'] ?? 0),
            TourAttributes::MINIMUM_ALTITUDE => (int) ($this->object['elevation']['minAltitude'] ?? 0),
            TourAttributes::PUBLIC_TRANSIT =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['publicTransit'] ?? '')),
            TourAttributes::PRIVATE_TRANSIT =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['gettingThere'] ?? '')),
            TourAttributes::STARTING_POINT =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['startingPointDescr'] ?? '')),
            TourAttributes::ENDING_POINT =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['tour']['destination'] ?? '')),
            TourAttributes::TECHNIQUE_RATING => (int) ($this->object['rating']['technique'] ?? 0),
            TourAttributes::TOUR_DURATION => (int) ($this->object['time']['min'] ?? 0),
            TourAttributes::TOUR_LENGTH => (int) ($this->object['length'] ?? 0),
            TourAttributes::DATA_SOURCE => TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE,
            TourAttributes::DATA_SOURCE_ID =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['id'] ?? '')),
            TourAttributes::DIRECTIONS =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['directions'] ?? '')),
            TourAttributes::PARKING =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['parking'] ?? '')),
            TourAttributes::ADDITIONAL_INFORMATION =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['additionalInformation'] ?? '')),
            TourAttributes::HINT_OF_AUTHOR =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['tip'] ?? '')),
            TourAttributes::SAFETY_NOTE =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['safetyGuidelines'] ?? '')),
            TourAttributes::LITERATURE_TIP_OF_AUTHOR =>
                $this->stringCleaner->cleanHtmlString((string) ($this->object['literature'] ?? '')),
        ];

        foreach ($simpleAttributes as $name => $data) {
            if (!is_string($data) || $data !== '') {
                $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
            }
        }

        $publicTransportFriendly = $this->object['publicTransportFriendly'] ?? null;
        if ($publicTransportFriendly === 'false') {
            $items[] = new AttributeAdapter(
                $this->getExternalId(),
                'labels',
                TourAttributes::PUBLIC_TRANSPORT_FRIENDLY
            );
        }

        $opened = $this->object['opened'] ?? null;
        if ($opened === false || $opened === 'false') {
            $items[] = new AttributeAdapter($this->getExternalId(), 'labels', TourAttributes::ISCLOSED);
        }

        return $items;
    }


    private function extractGeometry()
    {
        if ($this->cachedGeometry) {
            return $this->cachedGeometry;
        }

        $rawGeometry = $this->object['geometry'] ?? null;
        $geometry = [];
        if ($rawGeometry) {
            $geoItems = explode(' ', $rawGeometry);
            foreach ($geoItems as $geoItem) {
                $geo = explode(',', $geoItem);
                $geometry[] = [ (float) $geo[1], (float) $geo[0] ];
            }
        }

        $this->cachedGeometry = $geometry;
        return $geometry;
    }

    public function getSourceName(): ?string
    {
        $sourceName = $this->object['meta']['source']['name'] ?? null;
        return $sourceName ? (string) $sourceName : null;
    }

    public function getAuthorName(): ?string
    {
        return null;
    }

    public function getBookingUri(): ?string
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getCitySelectors(): array
    {
        $selectors = [];

        foreach ($this->object['regions']['region'] ?? [] as $region) {
            if (($region['type'] ?? null) === 'commune' && array_key_exists('id', $region)) {
                $selectors[] = new ExternalIdSelector(ExternalIdType::OUTDOORACTIVE, $region['id']);
            }
        }

        return $selectors;
    }
}
