<?php declare(strict_types=1);

namespace Newland\ToubizApi\Tests\Unit\Service\Tportal\ObjectAdapter;

use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\LodgingAdapter;
use PHPUnit\Framework\TestCase;

class LodgingAdapterTest extends TestCase
{

    // See TBPOI-377
    public function testUsesCompanyName(): void
    {
        $adapter = new LodgingAdapter([
            'housedata' => [
                'out' => [
                    'address' => [
                        'companyName1' => 'Foo'
                    ]
                ]
            ]
        ]);

        $this->assertEquals('Foo', $adapter->getName());
    }


    // See TBPOI-377
    public function testConcatenatesCompanyNamesIfMultipleAvailable(): void
    {
        $adapter = new LodgingAdapter([
              'housedata' => [
                  'out' => [
                      'address' => [
                          'companyName1' => 'Foo',
                          'companyName2' => 'Bar',
                          'companyName3' => 'Baz',
                      ]
                  ]
              ]
          ]);

        $this->assertEquals('Foo, Bar, Baz', $adapter->getName());
    }

    /** @dataProvider provideDescriptions */
    public function testProperlyStripDescription(string $expected, string $input): void
    {
        $adapter = new LodgingAdapter(
            [
                'housedata' => [
                    'out' => [
                        'descriptions' => [
                            'description' => $input
                        ]
                    ]
                ]
            ]
        );
        $this->assertEquals($expected, $adapter->getDescription());
    }

    public function provideDescriptions(): array
    {
        return [
            'valid text remains text' => ['text', 'text'],
            'invalid script is completely discarded' => ['', '<script>text</script>'],
            'invalid tag is removed' => ['text', '<div>text</div>'],
            'invalid custom tag is removed' => ['text', '<myTag>text</myTag>'],
            'invalid empty tag' => ['', '<u></u>'],
            'valid code stays unchanged' => ['<u>text</u>', '<u>text</u>'],
            'invalid post spacing is removed' => ['<u>text</u>', '<u >text</u>'],
            'invalid data attribute' => ['<u>text</u>', '<u data>text</u>'],
            'invalid data attribute with value' => ['<u>text</u>', '<u data="foo">text</u>'],
            'invalid attribute for tag (valid uri)' =>
                ['<u>text</u>', '<u href="www.example.com">text</u>'],
            'invalid data attribute followed by invalid attribute for tag' =>
                ['<u>text</u>', '<u href="www.example.com" data>text</u>'],
            'invalid attribute for tag followed by invalid data attribute' =>
                ['<u>text</u>', '<u data href="www.example.com">text</u>'],
            'invalid attribute for tag (title)' =>
                ['<u>text</u>', '<u title="Title">text</u>'],
            'invalid attribute for tag (target)' =>
                ['<u>text</u>', '<u target="_blank">text</u>'],
            'invalid a tag without attribute href' =>
                ['text', '<a>text</a>'],
            'valid a tag with href' =>
                ['<a href="www.example.com">text</a>', '<a href="www.example.com">text</a>'],
            'invalid a tag using single quotes' =>
                ['<a href="www.example.com">text</a>', '<a href=\'www.example.com\'>text</a>'],
            'invalid duplicate href' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" href="www.example-2.com">text</a>'],
            'invalid duplicate href with invalid data attribute in between' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" data href="www.example-2.com">text</a>'],
            'valid a tag with title' =>
                ['<a href="www.example.com" title="Title">text</a>',
                    '<a href="www.example.com" title="Title">text</a>'],
            'valid a tag with target' =>
                ['<a href="www.example.com" target="_blank" rel="noreferrer noopener">text</a>',
                    '<a href="www.example.com" target="_blank">text</a>'],
            'valid nesting is kept' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</u></strong>'],
            'invalid nesting (wrong order) is corrected' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</strong></u>'],
            'invalid nesting (missing closing tags) are added in the end' =>
                ['<strong><u><em>text</em></u></strong>', '<strong><u><em>text</em></strong>'],
            'invalid nesting (only closing tag) is removed' =>
                ['<strong><em>text</em></strong>', '<strong><em>text</em></u></strong>'],
        ];
    }


    public function testIgnoresMediaWithEmptySourceUri(): void
    {
        $adapter = new LodgingAdapter([
              'housedata' => [
                  'out' => [
                      'ID' => 'test',
                      'multimediaURLs' => [
                          [
                              'type' => [ 'value' => 'Preview' ],
                              'ID' => 'test',
                              'url' => '',
                          ],
                          [
                              'type' => [ 'value' => 'Preview' ],
                              'ID' => 'test',
                              'url' => 'https://placehold.it/500x500',
                          ],
                      ]
                  ]
              ]
          ]);

        $this->assertCount(1, $adapter->getMedia());
        $this->assertEquals('https://placehold.it/500x500', $adapter->getMedia()[0]->getSourceUri());
    }

}
