<?php
namespace Newland\Toubiz\Api\Service\Outdooractive;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Pool;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Newland\Toubiz\Api\Exception\InvalidServiceResponseException;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter\TourAdapter;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\StringCleaner;
use Psr\Log\LoggerAwareTrait;
use function Safe\json_decode;

/**
 * Service for Outdooractive API.
 */
class ApiService extends AbstractService
{
    use LanguageAware,
        Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://www.outdooractive.com/api/project/';

    /**
     * Path for fetching the tours list.
     * NOTE: `lastModifiedAfter` is added so the `lastModified` attribute is added to the response.
     *       It is not however set to the current delta: That is resolved in the `fetchIdListWithinCurrentDelta`
     *       method.
     * @var string
     */
    const PATH_TOURS_LIST = ':clientName/tours?key=:apiKey&lang=:language&lastModifiedAfter=1970-01-01';

    /**
     * @var string Path for a tour.
     */
    const PATH_TOUR = ':clientName/oois/:ids?key=:apiKey&lang=:language';

    const RECORDS_PER_DETAIL_REQUEST = 10;

    public function fetchTours(callable $block): ServiceResult
    {
        $all = $this->fetchTourList();
        if ($all === null) {
            throw new InvalidServiceResponseException('Could not fetch list of all tours');
        }

        if ($this->limit) {
            $all = array_splice($all, 0, $this->limit);
        }

        $ids = $this->getIdsWithinDelta($all);
        $requests = $this->buildRequestsForIds($ids);

        $total = \count($ids);
        (new Pool($this->httpClient, $requests, [
            'concurrency' => $this->parameters['concurrency'] ?? 10,
            'fulfilled' => function (Response $response, $index) use ($block, $total, $requests) {
                $data = $this->jsonResponse($response, $requests[$index]->getUri());
                foreach ($data['tour'] ?? [] as $tour) {
                    $block($this->tourDataToAdapter($tour), $total);
                }
            },
            'rejected' => function ($reason) {
                $this->logger->error($reason);
            }
        ]))->promise()->wait();

        $result = new ServiceResult();
        $result->setAll($this->listItemsToIds($all));
        return $result;
    }

    private function fetchTourList(): ?array
    {
        $response = $this->httpClient->send($this->prepareRequest(static::PATH_TOURS_LIST));

        if ($response->getStatusCode() !== 200) {
            return null;
        }

        $body = StringCleaner::asString((string) $response->getBody());
        $data = json_decode($body, true);
        return $data['data'] ?? null;
    }

    private function getIdsWithinDelta(array $tourList): array
    {
        $now = (new \DateTime())->getTimestamp();
        $within = array_filter(
            $tourList,
            function (array $item) use ($now) {
                $date = new \DateTime($item['lastModified']);
                return $date->add($this->delta)->getTimestamp() > $now;
            }
        );

        return $this->listItemsToIds($within);
    }

    /**
     * @param string[] $ids
     * @return Request[]
     */
    private function buildRequestsForIds(array $ids): array
    {
        $chunks = array_chunk($ids, static::RECORDS_PER_DETAIL_REQUEST);
        return array_map(
            function (array $ids) {
                return $this->prepareRequest(static::PATH_TOUR, [ ':ids' => implode(',', $ids) ]);
            },
            $chunks
        );
    }

    /**
     * @param array $listItems
     * @return string[]
     */
    private function listItemsToIds(array $listItems): array
    {
        return array_map(
            function (array $item) {
                return $item['id'];
            },
            $listItems
        );
    }

    protected function prepareRequest(string $uri, array $parameters = []): Request
    {
        $parameters = array_merge(
            $parameters,
            [
                ':apiKey' => $this->apiKey,
                ':clientName' => $this->clientName,
                ':language' => $this->language
            ]
        );

        $fullUri = str_replace(array_keys($parameters), array_values($parameters), $uri);

        return new Request('GET', $fullUri, [ 'accept' => 'application/json' ]);
    }

    private function tourDataToAdapter(array $tour): TourAdapter
    {
        $adapter = new TourAdapter($tour);
        if ($this->language !== null) {
            $adapter->setLanguage($this->language);
        }
        return $adapter;
    }
}
