<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Service\Toubiz\Legacy\TourApiService;

use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\TourApiService\TourAdapter;
use PHPUnit\Framework\TestCase;
use Psr\Log\Test\TestLogger;

class TourAdapterTest extends TestCase
{

    /** @dataProvider provideValidGpxFiles */
    public function testGetsGeometryFromGpxFile(string $file): void
    {
        [ $tour ] = $this->adapterForGpsFile($file);

        $attributes = $tour->getAttributes();
        $attributeNames = array_map(function(AttributeAdapterInterface $attribute) {
            return $attribute->getName();
        }, $attributes);
        $this->assertContains(TourAttributes::GEOMETRY, $attributeNames, 'Attributes should include geometry');
    }

    /** @dataProvider provideErroneousGpxFiles */
    public function testErrorWhenParsingGpxFileDoesNotCompletelyAbortExecution(string $file): void
    {
        [ $tour, $logger ] = $this->adapterForGpsFile($file);
        $attributes = $tour->getAttributes();
        $this->assertArrayNotHasKey(TourAttributes::GEOMETRY, $attributes);

        $messages = array_map(function(array $record) {
            return $record['message'];
        }, $logger->records);
        $this->assertContains('Error parsing GPS File', implode('|', $messages));
    }

    /** @dataProvider provideInvalidGpxFiles */
    public function testInvalidOrEmptyGpxFilesDoNotAddGeometry(string $file): void
    {
        [ $tour ] = $this->adapterForGpsFile($file);

        $attributes = $tour->getAttributes();
        $this->assertArrayNotHasKey(TourAttributes::GEOMETRY, $attributes);
    }

    public function provideErroneousGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/erroneous-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return $data;
    }

    public function provideValidGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/valid-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return $data;
    }

    public function provideInvalidGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/invalid-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return array_merge($data, $this->provideErroneousGpxFiles());
    }

    private function adapterForGpsFile(string $file): array
    {
        $tour = new TourAdapter([
            'remote_id' => 'TEST__' . preg_replace('/\W/', '-', $file),
            'object' => [
                'data_map' => [
                    'geodata_file_gpx' => [
                        'content' => [
                            'path' => $file
                        ]
                    ]
                ]
            ]
        ]);

        $logger = new TestLogger();
        $tour->setLogger($logger);

        return [ $tour, $logger ];
    }

    /** @dataProvider provideDescriptions */
    public function testProperlyStripDescription(string $expected, string $input): void
    {
        $adapter = new TourAdapter(
            [
                'object' => [
                    'data_map' => [
                        'description' => [
                            'content' => $input
                        ]
                    ]
                ]
            ]
        );
        $this->assertEquals($expected, $adapter->getDescription());
    }

    public function provideDescriptions(): array
    {
        return [
            'valid text remains text' => ['text', 'text'],
            'invalid script is completely discarded' => ['', '<script>text</script>'],
            'invalid tag is removed' => ['text', '<div>text</div>'],
            'invalid custom tag is removed' => ['text', '<myTag>text</myTag>'],
            'invalid empty tag' => ['', '<u></u>'],
            'valid code stays unchanged' => ['<u>text</u>', '<u>text</u>'],
            'invalid post spacing is removed' => ['<u>text</u>', '<u >text</u>'],
            'invalid data attribute' => ['<u>text</u>', '<u data>text</u>'],
            'invalid data attribute with value' => ['<u>text</u>', '<u data="foo">text</u>'],
            'invalid attribute for tag (valid uri)' =>
                ['<u>text</u>', '<u href="www.example.com">text</u>'],
            'invalid data attribute followed by invalid attribute for tag' =>
                ['<u>text</u>', '<u href="www.example.com" data>text</u>'],
            'invalid attribute for tag followed by invalid data attribute' =>
                ['<u>text</u>', '<u data href="www.example.com">text</u>'],
            'invalid attribute for tag (title)' =>
                ['<u>text</u>', '<u title="Title">text</u>'],
            'invalid attribute for tag (target)' =>
                ['<u>text</u>', '<u target="_blank">text</u>'],
            'invalid a tag without attribute href' =>
                ['text', '<a>text</a>'],
            'valid a tag with href' =>
                ['<a href="www.example.com">text</a>', '<a href="www.example.com">text</a>'],
            'invalid a tag using single quotes' =>
                ['<a href="www.example.com">text</a>', '<a href=\'www.example.com\'>text</a>'],
            'invalid duplicate href' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" href="www.example-2.com">text</a>'],
            'invalid duplicate href with invalid data attribute in between' =>
                ['<a href="www.example-1.com">text</a>',
                    '<a href="www.example-1.com" data href="www.example-2.com">text</a>'],
            'valid a tag with title' =>
                ['<a href="www.example.com" title="Title">text</a>',
                    '<a href="www.example.com" title="Title">text</a>'],
            'valid a tag with target' =>
                ['<a href="www.example.com" target="_blank" rel="noreferrer noopener">text</a>',
                    '<a href="www.example.com" target="_blank">text</a>'],
            'valid nesting is kept' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</u></strong>'],
            'invalid nesting (wrong order) is corrected' =>
                ['<strong><u>text</u></strong>', '<strong><u>text</strong></u>'],
            'invalid nesting (missing closing tags) are added in the end' =>
                ['<strong><u><em>text</em></u></strong>', '<strong><u><em>text</em></strong>'],
            'invalid nesting (only closing tag) is removed' =>
                ['<strong><em>text</em></strong>', '<strong><em>text</em></u></strong>'],
        ];
    }
}
