<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\Promise\RejectionException;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Exception\InvalidJsonException;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMap;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMapParser;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;
use Newland\Toubiz\Api\Utility\RetryPool;
use Psr\Log\LoggerAwareTrait;

/**
 * Service for legacy Toubiz DB API.
 */
class DbService extends AbstractService
{
    use LanguageAware;
    use Limitable;
    use LoggerAwareTrait;
    use SharedLegacyToubizServiceBehaviour;

    const DEFAULT_BASE_URI = 'https://db-service.toubiz.de/';

    public function fetchPointOfInterests(callable $block): ServiceResult
    {
        $map = $this->getCategoryMap();
        $concurrency = $this->parameters['concurrency'] ?? 10;
        $perPage = 25;
        $pool = new ConcurrentPaginatedRequests($concurrency, function (int $page) use ($block, $perPage, $map) {
            if (!$this->withinLimit($page * $perPage)) {
                return new RejectionException('Page is outside of limit');
            }
            return $this->sendRequest($page, $perPage)->then(function (array $data) use ($map, $block) {
                foreach ($data['children_list'] as $item) {
                    $block($this->poiDataToAdapter($item, $map), $this->limit ?? $data['children_count'] ?? null);
                }
            });
        });

        $pool->start()->wait();

        // TODO fetch deleted records
        return new ServiceResult();
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $page
     * @param int $limit
     * @param int $retriesIfFail
     * @return PromiseInterface
     */
    protected function sendRequest(int $page, int $limit, int $retriesIfFail = 5): PromiseInterface
    {
        $offset = ($page - 1) * $limit;
        $url = $this->buildUrl([
            ':apiKey' => $this->apiKey,
            ':clientName' => $this->clientName,
            'class' => 'attraction',
            'offset' => $offset,
            'limit' => $limit,
            'modified' => $this->delta->asSeconds(),
        ]);

        return $this->jsonRequest(new Uri($url), $retriesIfFail, function (array $data) {
            return $data['content'] && count($data['content']['children_list']) > 0;
        })->then(function (array $data) {
            return $data['content'];
        });
    }

    private function poiDataToAdapter(array $item, CategoryMap $categoryMap): PointOfInterestAdapter
    {
        $article = new PointOfInterestAdapter($item);
        $article->setCategoryMap($categoryMap);
        if ($this->language) {
            $article->setLanguage($this->language);
        }
        return $article;
    }
}
