<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Exception\InvalidJsonException;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMap;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMapParser;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\CityApiService\CityAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;
use Newland\Toubiz\Api\Utility\RetryPool;
use Psr\Log\LoggerAwareTrait;

/**
 * Service for legacy Toubiz DB API.
 */
class CityApiService extends AbstractService
{
    use LanguageAware;
    use Limitable;
    use LoggerAwareTrait;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://db-service.toubiz.de/';

    /**
     * Map of language constants to URI segments expected by the API.
     *
     * @var array
     */
    private static $languageMap = [
        Language::DE => 'ger-DE',
        Language::EN => 'eng-GB',
        Language::FR => 'fre-FR',
        Language::ES => 'esl-ES',
    ];

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [
        'Article' => PointOfInterestAdapter::class,
    ];
    const CATEGORY_MAP_CSV = '/../../../../Resources/Service/Toubiz/Legacy/DbService/categories-poi.csv';

    /**
     * Fetch records.
     *
     * @param callable $block
     * @return void
     */
    public function fetchCities(callable $block): void
    {
        $offset = 0;
        $limit = 25;
        while ($this->withinLimit($offset + $limit)) {
            $data = $this->sendRequest($offset, $limit);
            if (!$data) {
                return;
            }

            foreach ($data as $item) {
                $article = new CityAdapter($item);
                if ($this->language) {
                    $article->setLanguage($this->language);
                }
                $block($article);
            }

            $offset += $limit;
        }
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $offset
     * @param int $limit
     * @param int $retriesIfFail
     * @return mixed
     */
    protected function sendRequest($offset, $limit, $retriesIfFail = 5)
    {
        $parameters = [
            ':apiKey' => $this->apiKey,
            ':clientName' => $this->clientName,
            ':offset' => $offset,
            ':limit' => $limit,
        ];

        $url = str_replace(
            array_keys($parameters),
            $parameters,
            '/layout/set/json/content/view/json/:clientName/' .
            '(key)/:apiKey/(class)/city/(modified)/864000000/(offset)/:offset/(limit)/:limit'
        );
        $url = $this->appendLanguageToUrl($url);

        $pool = new RetryPool($retriesIfFail);
        $pool->setLogger($this->logger);
        $response = $pool->retryOnException(function () use ($url) {
            return $this->httpClient->request('GET', $url);
        });
        if ($response->getStatusCode() === 200) {
            $dataRaw = $response->getBody();
            $dataRaw = StringCleaner::asString($dataRaw);
            $data = json_decode($dataRaw, true);

            if ($data === null) {
                throw new InvalidJsonException(
                    sprintf('API request returned invalid JSON (offset %d, limit %d).', $offset, $limit),
                    1557149751
                );
            }

            if ($data['content'] && count($data['content']['children_list']) > 0) {
                return $data['content']['children_list'];
            }
        }

        return false;
    }

    /**
     * The category map is built from a static csv export from Toubiz Legacy.
     * This should be updated when new categories are added in Toubiz Legacy Backend.
     */
    private function getCategoryMap(): CategoryMap
    {
        $file = __DIR__ . self::CATEGORY_MAP_CSV;

        return (new CategoryMapParser())->parse($file, (string) $this->language);
    }

    private function appendLanguageToUrl(string $url): string
    {
        $languageUrlSegment = static::$languageMap[$this->language] ?? null;
        if ($languageUrlSegment) {
            return sprintf('%s/(lang)/%s', $url, $languageUrlSegment);
        }
        return $url;
    }
}
