<?php
namespace Newland\Toubiz\Api\Service\Tportal;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\LodgingAdapter;

/**
 * Service for legacy Toubiz TPortal.
 *
 * Concrete implementation to communicate with the TPortal
 * which is providing data for TOMAS-bound entities.
 */
class ApiService extends AbstractService
{
    /**
     * @var string Base URI of API endpoint.
     */
    const BASE_URI = 'https://tportal.toubiz.de';

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [
        'Lodging' => LodgingAdapter::class
    ];

    /**
     * Fetch all lodgings.
     *
     * @return array<LodgingAdapter>
     */
    public function fetchLodgings()
    {
        $data = $this->fetchInBadges();
        if (!$data) {
            return false;
        }

        $articles = [];
        foreach ($data as $item) {
            // Injecting the source as it is required for building links.
            $item['clientName'] = $this->clientName;
            $articles[] = (new LodgingAdapter($item));
        }
        return $articles;
    }

    /**
     * Fetches records in badges.
     *
     * This is due to the API returning an emtpy response when
     * too many records are present.
     *
     * @todo Implement logic in abstract service.
     * @return mixed
     */
    protected function fetchInBadges()
    {
        $data = [];
        $page = 1;
        while (true) {
            $response = $this->sendRequest($page);
            if (!$response) {
                return $data;
            }

            $data = array_merge($data, $response);
            $page++;
        }
        return $data;
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $page
     * @return array|bool
     */
    protected function sendRequest($page)
    {
        $parameters = [
            ':clientName' => $this->clientName,
            ':page' => $page
        ];

        $url = str_replace(
            array_keys($parameters),
            array_values($parameters),
            '/:clientName/ukv/search?reset=1&json=1&page=:page'
        );

        $response = $this->httpClient->request('GET', $url);
        if ($response->getStatusCode() == 200) {
            $data = json_decode($response->getBody(), true);
            if ($data['housedata'] && count($data['housedata']) > 0) {
                return $data['housedata'];
            } else {
                return false;
            }
        } else {
            return false;
        }
    }
}
