<?php
namespace Newland\Toubiz\Api\Service\Sisag;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\Sisag\ObjectAdapter\RegionAdapter;
use Newland\Toubiz\Api\Service\Sisag\ObjectAdapter\SlopeAdapter;
use Newland\Toubiz\Api\Service\Sisag\ObjectAdapter\TransportationFacilityAdapter;

/**
 * Service for SISAG API.
 *
 * Concrete implementation to communicate with
 * the SISAG API.
 */
class ApiService extends AbstractService
{
    /**
     * @var string Base URI of API endpoint.
     */
    const BASE_URI = 'http://siscontrol.sisag.ch/sismedia/:clientName:/SisControlAktuell.xml';

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [];

    /**
     * @var array Internal cache.
     */
    protected $cache = [];

    /**
     * Fetch regions.
     *
     * @return array<RegionAdapter>
     */
    public function fetchRegions()
    {
        $data = $this->fetchRawReport();
        $collection = [];
        foreach ($data->REGIONS->children() as $item) {
            $this->injectUpdatedAt($item, $data);
            $collection[] = (new RegionAdapter($item));
        }
        return $collection;
    }

    /**
     * Fetch transportation facilities.
     *
     * @return array<TransportationFacilityAdapter>
     */
    public function fetchTransportationFacilities()
    {
        $data = $this->fetchRawReport();
        $collection = [];
        foreach ($data->LIFTE->children() as $item) {
            /*
             * A status of "0" indicates an inactive / non-visible
             * facility which should not be imported at all.
             */
            if ((int)$item->STATUS === 0) {
                continue;
            }

            $this->injectUpdatedAt($item, $data);
            $collection[] = (new TransportationFacilityAdapter($item));
        }
        return $collection;
    }

    /**
     * Fetch slopes.
     *
     * @return array<SlopeAdapter>
     */
    public function fetchSlopes()
    {
        $data = $this->fetchRawReport();
        $collection = [];
        foreach ($data->PISTEN->children() as $item) {
            $this->injectUpdatedAt($item, $data);
            $collection[] = (new SlopeAdapter($item));
        }
        return $collection;
    }

    /**
     * Fetch report.
     *
     * The "API" basically is an XML file that is being generated
     * in regular intervals.
     *
     * @return array
     * @throws \Exception if no data has been received from the API
     */
    public function fetchRawReport()
    {
        if (array_key_exists('fetchRawReport', $this->cache)) {
            return $this->cache['fetchRawReport'];
        }

        // Override already initialized http client.
        $baseUri = str_replace(':clientName:', $this->clientName, self::BASE_URI);
        $this->httpClient = new \GuzzleHttp\Client();
        $response = $this->httpClient->request('GET', $baseUri);

        if ($response->getStatusCode() == 200) {
            $this->cache['fetchRawReport'] = simplexml_load_string($response->getBody());
            return $this->cache['fetchRawReport'];
        } else {
            throw new \Exception('No data received for synchronizing!');
        }
    }

    /**
     * Adds a child node 'AUSGABE' to the given xml node
     * which represents the node from the parent object containing
     * the updated-at timestamp.
     *
     * @param \SimpleXMLElement &$xmlNode
     * @param \SimpleXMLElement $report
     * @return void
     */
    protected function injectUpdatedAt(\SimpleXMLElement &$xmlNode, \SimpleXMLElement $report)
    {
        $xmlNode->addChild('AUSGABE', $report->AUSGABE);
    }
}
