<?php
namespace Newland\PageFrameProvider\Routing;

use Neos\Flow\Annotations as Flow;
use Neos\Neos\Domain\Service\ContentDimensionPresetSourceInterface;

/**
 * Simple helper for routing related tasks in PageFrameProvider.
 * In contrast to `FrontendNodeRoutePartHandler` this class is not deeply coupled to the Neos
 * internals and thus is easier to test.
 *
 * @Flow\Scope("singleton")
 */
class RoutingHelper
{
    /**
     * Separator of dimensions in the URL portion. If multiple dimensions exist in the
     * URL then this string is used to separate them. This value must be the same as used
     * in the Neos core.
     * @see OriginalFrontendNodeRoutePartHandler::getUriSegmentForDimensions
     * @see https://neos.readthedocs.io/en/4.2/CreatingASite/ContentDimensions.html#routing
     */
    const DIMENSION_SEPARATOR = '_';

    /**
     * URL Separator: Separates path parts in a URL (e.g. foo/bar/baz are separated by a slash).
     */
    const URL_SEPARATOR = '/';

    /**
     * Custom delimiter for regex. Must not contain DIMENSION_SEPARATOR or URL_SEPARATOR or any value
     * that may be used in a dimensions uriSegment.
     */
    const REGEX_DELIMITER = '#';

    /**
     * @Flow\Inject
     * @var ContentDimensionPresetSourceInterface
     */
    protected $contentDimensionPresetSource;

    public function extractDimensionValuesFromPath(string $path): string
    {
        preg_match($this->validDimensionsRegex(), $path, $matches);
        return $matches[0] ?? '';
    }

    public function prepareUrlForFurtherUseInRoutingPipeline(string $url): string
    {
        $url = $this->stripDimensionValuesFromPath($url);

        // PageFrameProvider URLs are defined as `{node}/foo/bar`
        $url = $this->ensureLeadingSlashExists($url);

        return $url;
    }

    private function stripDimensionValuesFromPath(string $path): string
    {
        return (string) preg_replace($this->validDimensionsRegex(), '', $path);
    }

    private function ensureLeadingSlashExists(string $path): string
    {
        return static::URL_SEPARATOR . ltrim($path, static::URL_SEPARATOR);
    }

    private function validDimensionsRegex(): string
    {
        $dimensionParts = [];
        foreach ($this->contentDimensionPresetSource->getAllPresets() as $dimension) {
            $presetSegments = array_map(
                function (array $preset) {
                    return $preset['uriSegment'];
                },
                $dimension['presets']
            );
            $dimensionParts[] = '(' . implode('|', $presetSegments) . ')';
        }

        $regex = '^(' . implode(static::DIMENSION_SEPARATOR, $dimensionParts) . ')';
        return static::REGEX_DELIMITER . $regex . static::REGEX_DELIMITER;
    }
}
