<?php declare(strict_types=1);

/*
 * Copyright (c) 2023, land in sicht AG
 *
 * www.land-in-sicht.de - All rights reserved.
 *
 * This is proprietary software. Unauthorized copying
 * of this file, via any medium, is strictly prohibited.
 */

use Newland\PageFrameProvider\Definition\RouteDefinition;
use PHPUnit\Framework\TestCase;

class RouteDefinitionTest extends TestCase
{

    public function testParsesSimpleString(): void
    {
        $definition = new RouteDefinition('foo/bar');
        $this->assertEquals([ '_fallback' => 'foo/bar' ], $definition->getUriSegments());
    }

    public function testParsesSimpleStringInUriSegmentConfig(): void
    {
        $definition = new RouteDefinition([ 'uriSegment' => 'foo/bar' ]);
        $this->assertEquals([ '_fallback' => 'foo/bar' ], $definition->getUriSegments());
    }

    public function testParsesDeepConfigurationInUriSegmentConfig(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => [
                    'de' => 'foo/bar',
                ],
            ]
        );
        $this->assertEquals([ 'de' => 'foo/bar' ], $definition->getUriSegments());
    }

    public function testParsesBreadcrumbsSimpleString(): void
    {
        $definition = new RouteDefinition('foo/bar');
        $this->assertEquals([], $definition->getBreadcrumb());
    }

    public function testParsesBreadcrumbsIfNotInArray(): void
    {
        $definition = new RouteDefinition([ 'uriSegment' => 'foo/bar' ]);
        $this->assertEquals([], $definition->getBreadcrumb());
    }

    public function testParsesBreadcrumbsIfAvailable(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => 'foo/bar',
                'breadcrumb' => [ 'foo', 'bar', 'baz' ],
            ]
        );
        $this->assertEquals([ 'foo', 'bar', 'baz' ], $definition->getBreadcrumb());
    }

    public function testFetchesBreadcrumbsForGivenSiteIfAvailable(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => 'foo/bar',
                'breadcrumb' => [
                    'site-test' => [ 'site', 'test' ],
                ],
            ]
        );

        $this->assertEquals([ 'site', 'test' ], $definition->getBreadcrumb('site-test'));
    }

    public function testParsesFallbackBreadcrumbIfAvailable(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => 'foo/bar',
                'breadcrumb' => [
                    '_fallback' => [ 'fallback', 'test' ],
                    'another-site' => [ 'another', 'site', 'test' ],
                ],
            ]
        );

        $this->assertEquals([ 'fallback', 'test' ], $definition->getBreadcrumb('site-test'));
    }

    public function testUsesOldStyleBreadcrumbConfigurationAsFallback(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => 'foo/bar',
                'breadcrumb' => [
                    'old',
                    'simple',
                    'array',
                    'site-test' => [ 'site', 'test' ],
                ],
            ]
        );

        $this->assertEquals([ 'site', 'test' ], $definition->getBreadcrumb('site-test'));
        $this->assertEquals([ 'old', 'simple', 'array' ], $definition->getBreadcrumb('site-unknown'));
    }

    public function testFetchesUriSegmentForLanguage(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => [
                    'de' => 'foo/bar',
                ],
            ]
        );
        $this->assertEquals('foo/bar', $definition->getUriSegment('de'));
    }

    public function testUsesFallbackSegmentIfLanguageNotAvailable(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => [
                    '_fallback' => 'fallback/uri',
                ],
            ]
        );
        $this->assertEquals('fallback/uri', $definition->getUriSegment('ru'));
    }

    public function testStripsLeadingAndTrailingSlashesFromUris(): void
    {
        $definition = new RouteDefinition(
            [
                'uriSegment' => [
                    'de' => '/has/leading/and/trailing/slashes/',
                ],
            ]
        );
        $this->assertEquals(
            'has/leading/and/trailing/slashes',
            $definition->getUriSegment('de')
        );
    }
}
