<?php
namespace Newland\PageFrameProvider\Routing;

use Generator;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Aop\AdvicesTrait;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Neos\Flow\Mvc\Routing\Dto\MatchResult;
use Neos\Flow\Mvc\Routing\Dto\ResolveResult;
use Neos\Flow\ObjectManagement\Exception\InvalidObjectException;
use Neos\Flow\ObjectManagement\ObjectManager;
use \Neos\Neos\Routing\FrontendNodeRoutePartHandler as OriginalFrontendNodeRoutePartHandler;
use Neos\Flow\Annotations as Flow;
use Neos\Utility\ObjectAccess;
use Newland\PageFrameProvider\NodeResolution\RootNodeResolver;
use Newland\PageFrameProvider\Service\PageFrameContextService;

/**
 * Route Part Handler that defines the routing core for page frame provider.
 * It wraps Neos' core FrontendNodeRoutePart and enhances it by using 2 configurable
 * handlers:
 *
 * - RootNodeResolver: Defines which node will be used to base the page frame rendering on.
 *   Not only does this define how rendering looks like, but also which part of the URl matches
 *   the node and which part is used by a UriResolver. It is configurable using the
 *   `Newland.PageFrameProvider.nodeResolution` setting and defaults to only using a sites root node.
 *
 * - UriResolver: Defines additional, route specific parts of the URL that are matched in addition
 *    to the node parts. Each page frame provider route **must** be defined with a UriResolver using
 *    the `uriResolver` route part option.
 *
 * @example
 * # A page frame provider route can be defines as follows:
 * - name: 'Newland.Toubiz.Events.Neos:Events.Show'
 *   uriPattern: '{node}/{--plugin.event}'
 *   appendExceedingArguments: true
 *   defaults:
 *      '@package': 'newland.pageframeprovider'
 *      '@controller': 'pageframe'
 *      '@action': 'show'
 *      '@format': 'html'
 *      '--plugin':
 *          '@package': 'newland.toubiz.events.neos'
 *          '@controller': 'events'
 *          '@action': 'show'
 *          '@format': 'html'
 *   routeParts:
 *      node:
 *          handler: 'Newland\PageFrameProvider\Routing\FrontendNodeRoutePartHandler'
 *          options:
 *              uriResolver: 'Newland\Toubiz\Events\Neos\PageFrame\EventsUriResolver'
 *   '--plugin.event':
 *      objectType: 'Newland\Toubiz\Sync\Neos\Domain\Model\Event'
 *      uriPattern: '{title}-{urlIdentifier}'
 */
class FrontendNodeRoutePartHandler extends CompositeRouteHandler
{
    use AdvicesTrait;

    /**
     * @var PageFrameContextService
     * @Flow\Inject()
     */
    protected $pageFrameContext;

    /**
     * @var RootNodeResolver
     * @Flow\Inject()
     */
    protected $rootNodeResolver;

    /**
     * @var ObjectManager
     * @Flow\Inject()
     */
    protected $objectManager;

    public function injectCompositeRouteHandler(ObjectManager $objectManager): void
    {
        $this->compositeRouteHandler = $objectManager->get(OriginalFrontendNodeRoutePartHandler::class);
    }

    /**
     * Checks whether this Route Part corresponds to the given $routePath.
     * This method does not only check if the Route Part matches. It can also
     * shorten the $routePath by the matching substring when matching is successful.
     * This is why $routePath has to be passed by reference.
     *
     * @param string &$routePath The request path to be matched - without query parameters, host and fragment.
     * @return bool|MatchResult true or an instance of MatchResult if Route Part matched $routePath, otherwise false.
     */
    public function match(&$routePath)
    {
        $uriResolver = $this->uriResolver();

        foreach ($this->findAllowedNodesOnRequestPath($routePath) as [$nodeRequestPath, $nodeContextPath ]) {
            $potentiallyDomainSpecificPath = substr($routePath, strlen($nodeRequestPath));
            $domainSpecificPath = $uriResolver->getMatchingDomainSpecificPath($potentiallyDomainSpecificPath);
            if ($domainSpecificPath !== null) {
                $matchedPath = sprintf('%s/%s', $nodeRequestPath, $domainSpecificPath);
                $matchedPath = trim($matchedPath, '/');
                $routePath = str_replace($matchedPath, '', $routePath);
                return new MatchResult($nodeContextPath);
            }
        }

        return false;
    }

    /**
     * Checks whether this Route Part corresponds to the given $routeValues.
     * This method does not only check if the Route Part matches. It also
     * removes resolved elements from $routeValues-Array.
     * This is why $routeValues has to be passed by reference.
     *
     * @param array &$routeValues An array with key/value pairs to be resolved by Dynamic Route Parts.
     * @return bool|ResolveResult true or an instance of ResolveResult if Route Part can resolve one or more $routeValues elements, otherwise false.
     */
    public function resolve(array &$routeValues)
    {
        $node = ObjectAccess::getPropertyPath($routeValues, $this->name);
        if (!($node instanceof NodeInterface)) {
            return false;
        }

        $resolveResult = $this->resolveCompositeRouteHandler($routeValues);
        if ($resolveResult === false) {
            return false;
        }

        $this->pageFrameContext->setDimensions($node->getContext()->getTargetDimensionValues());
        $domainSpecificUriPath = $this->uriResolver()->resolveDomainSpecificPath($routeValues, $node);
        if ($domainSpecificUriPath === null) {
            return false;
        }

        if (array_key_exists('pageFrame', $routeValues)) {
            unset($routeValues['pageFrame']);
        }

        $resolvedPath = sprintf('%s/%s', $resolveResult->getResolvedValue(), $domainSpecificUriPath);
        $resolvedPath = trim($resolvedPath, '/');
        return new ResolveResult($resolvedPath);
    }

    /**
     * Every part of request path could be part of the node or more routing
     * definitions. This method returns a generator that yields a path if
     * - It finds a node under that path.
     * - The current node resolver allows that path as a page frame base.
     *
     * This means that a request path such as `/foo/bar/baz` will yield at most
     * 4 different request node paths:
     * - `foo/bar/baz`
     * - `foo/bar`
     * - `foo`
     * - (empty string)
     *
     * @return Generator<array>
     */
    private function findAllowedNodesOnRequestPath(string $requestPath): Generator
    {
        $splitString = '/';
        $parts = (array) explode($splitString, $requestPath);
        while (\count($parts) > 0) {

            $pathToMatch = implode($splitString, $parts);
            $compositeResult = $this->matchCompositeRouteHandler($pathToMatch);
            if ($compositeResult instanceof MatchResult) {
                $nodePath = (string) $compositeResult->getMatchedValue();
                if ($this->rootNodeResolver->isNodePathAllowed($nodePath)) {
                    yield [ $pathToMatch, $nodePath ];
                }
            }

            array_pop($parts);
        }

        $routePath = '';
        $compositeResult = $this->matchCompositeRouteHandler($routePath);
        if ($compositeResult instanceof MatchResult) {
            $nodePath = (string) $compositeResult->getMatchedValue();
            if ($this->rootNodeResolver->isNodePathAllowed($nodePath)) {
                yield [ '', $nodePath ];
            }
        }
    }

    private function uriResolver(): UriResolver
    {
        if (!array_key_exists('uriResolver', $this->options)) {
            throw new InvalidConfigurationException(sprintf(
                'RoutePart %s requires the `uriResolver` option to be set',
                $this->name
            ));
        }

        $resolver = $this->objectManager->get($this->options['uriResolver']);
        if (!($resolver instanceof UriResolver)) {
            throw new InvalidObjectException(sprintf(
                'uriResolvers must implement %s but %s does not.',
                UriResolver::class,
                $this->options['uriResolver']
            ));
        }

        return $resolver;
    }
}
