<?php
namespace Newland\PageFrameProvider\Service;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Routing\UriBuilder;
use Newland\PageFrameProvider\Controller\PageFrameController;
use Newland\PageFrameProvider\NodeResolution\RootNodeResolver;
use function Safe\preg_replace;

/**
 * Simple service that allows linking to a page frame plugin controller.
 *
 * @see PageFrameController
 * @Flow\Scope("singleton")
 */
class PageFrameLinkingService
{
    const PACKAGE = 'newland.pageframeprovider';
    const CONTROLLER = 'pageframe';
    const ACTION = 'show';
    const DEFAULT_ARGUMENT_NAMESPACE = '--plugin';
    const DEFAULT_PAGE_FRAME = 'default';

    /**
     * @var RootNodeResolver
     * @Flow\Inject()
     */
    protected $nodeResolver;

    /**
     * Builds a new link to a plugin inside of a page frame container.
     * This requires the given plugin to be registered as a PageFrame route (see README.md).
     *
     * @param UriBuilder $uriBuilder
     * @param string $package
     * @param string $controller
     * @param string $action
     * @param string $pageFrame
     * @param array $arguments
     * @param string $argumentNamespace
     * @param NodeInterface $referenceNode
     * @return string
     */
    public function build(
        UriBuilder $uriBuilder,
        string $package,
        string $controller,
        string $action,
        string $pageFrame = self::DEFAULT_PAGE_FRAME,
        array $arguments = [],
        string $argumentNamespace = self::DEFAULT_ARGUMENT_NAMESPACE,
        NodeInterface $referenceNode = null,
        bool $absolute = false
    ): string {
        // UriBuilder is cloned to prevent modifications to the existing instance leaking
        // out of this Service.
        $uriBuilder = clone $uriBuilder;

        if (!$referenceNode) {
            /** @var NodeInterface $referenceNode */
            $referenceNode = $uriBuilder->getRequest()->getInternalArgument('__node');
        }

        $url = $this->preventUriBuilderFromAddingCurrentParametersToGeneratedUrl($uriBuilder)
            ->reset()
            ->setCreateAbsoluteUri($absolute)
            ->build([
                'node' => $this->nodeResolver->resolveRootNode($referenceNode),
                '@package' => static::PACKAGE,
                '@controller' => static::CONTROLLER,
                '@action' => static::ACTION,
                'pageFrame' => $pageFrame,
                $argumentNamespace => array_replace_recursive([
                    '@package' => strtolower($package),
                    '@controller' => strtolower($controller),
                    '@action' => strtolower($action),
                ], $arguments),
            ]);

        return $this->cleanUrl($url);
    }

    /**
     * Checks whether or not the given route values correspond to a page frame
     * link to the given plugin.
     *
     * @param array $routeValues
     * @param string $package
     * @param string $controller
     * @param string $action
     * @param string $argumentNamespace
     * @return bool
     */
    public function isPageFrameActionLink(
        array $routeValues,
        string $package,
        string $controller,
        string $action,
        string $argumentNamespace = self::DEFAULT_ARGUMENT_NAMESPACE
    ): bool {
        return ($routeValues['@package'] ?? null) === static::PACKAGE &&
            ($routeValues['@controller'] ?? null) === static::CONTROLLER &&
            ($routeValues['@action'] ?? null) === static::ACTION &&
            ($routeValues[$argumentNamespace]['@package'] ?? null) === $package &&
            ($routeValues[$argumentNamespace]['@controller'] ?? null) === $controller &&
            ($routeValues[$argumentNamespace]['@action'] ?? null) === $action;
    }

    /**
     * Prevents the given uri builder from adding the current request parameters to the
     * URL. This effectively disables `UriBuilder::mergeArgumentsWithRequestArguments`.
     *
     * @param UriBuilder $uriBuilder
     * @return UriBuilder
     */
    private function preventUriBuilderFromAddingCurrentParametersToGeneratedUrl(UriBuilder $uriBuilder): UriBuilder
    {
        $uriBuilder->setAddQueryString(false);
        $uriBuilder->setRequest($uriBuilder->getRequest()->getMainRequest());
        $uriBuilder->setArgumentsToBeExcludedFromQueryString(array_keys($uriBuilder->getRequest()->getArguments()));
        return $uriBuilder;
    }

    /**
     * Removes internal arguments (those that start with an underscore) from the final URL
     * as these are only for internal routing purposes and have no functional value
     * to the application and cleans the URL up in general.
     *
     * @param string $url
     * @return string
     */
    public function cleanUrl(string $url): string
    {
        // Every argument that starts with --plugin[_*] is removed
        $uri = preg_replace('/[&?]--plugin(\[|%5B)_.*?(\]|%5D)=.*(?=(&|$))/', '', $url);

        // Multiple slashes in a row (e.g. foo//bar) are replaced with a single one.
        // Except if preceded by a colon (as in http://).
        /** @var string $uri */
        $uri = preg_replace('/(?<!:)\/\/+/', '/', $uri);
        return $uri;
    }
}
