<?php declare(strict_types=1);

/*
 * Copyright (c) 2023, land in sicht AG
 *
 * www.land-in-sicht.de - All rights reserved.
 *
 * This is proprietary software. Unauthorized copying
 * of this file, via any medium, is strictly prohibited.
 */

namespace Newland\PageFrameProvider\Definition;

use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Annotations as Flow;
use Newland\PageFrameProvider\Service\PageFrameContextService;

class RoutingConfigurationHelper
{
    /**
     * @var ConfigurationManager
     * @Flow\Inject()
     */
    protected $configurationManager;

    /**
     * @var RouteDefinition[]
     */
    protected $definitions;

    /**
     * @var PageFrameContextService
     * @Flow\Inject
     */
    protected $pageFrameContext;

    /**
     * @var array
     */
    protected $typeConfiguration = [];

    public function setTypeConfiguration(array $typeConfiguration): void
    {
        $this->typeConfiguration = $typeConfiguration;
    }

    /**
     * @return RouteDefinition[]
     */
    public function getAllDefinitions(): array
    {
        return $this->ensureDefinitionsExist();
    }

    public function getDefinition(?string $type = null): ?RouteDefinition
    {
        $this->ensureDefinitionsExist();
        return $this->definitions[$type] ?? null;
    }

    public function urlFragmentForType(?string $type = null): ?string
    {
        $definition = $this->getDefinition($type) ?? $this->getDefinition('_fallback');

        if (!$definition) {
            return null;
        }

        $url = $definition->getUriSegment($this->language());
        if (!$url) {
            return null;
        }

        return trim($url, '/');
    }

    public function removeUriSegmentsFromRoutePath(string $routePath): string
    {
        $language = $this->language();
        $matchGroup = implode(
            '|',
            array_map(
                function (string $segment) {
                    return preg_quote($segment, '/');
                },
                $this->getSortedUriSegmentsForLanguage($language)
            )
        );

        $regex = sprintf('/^\/?(%s)/', $matchGroup);
        return (string) preg_replace($regex, '', $routePath);
    }

    /**
     * Returns the possible uri segments for the given language sorted to have the longest segment
     * as the first item and the shortest segment as the last item in order to prevent conflicts when
     * searching for them in a path.
     *
     * @return string[]
     */
    private function getSortedUriSegmentsForLanguage(?string $language): array
    {
        $uriSegments = array_filter(
            array_map(
                static fn (RouteDefinition $definition) => $definition->getUriSegment($language),
                array_values($this->getAllDefinitions())
            )
        );

        usort(
            $uriSegments,
            static fn (string $a, string $b) => strlen($b) - strlen($a)
        );

        return $uriSegments;
    }

    /**
     * @return RouteDefinition[]
     */
    private function ensureDefinitionsExist(): array
    {
        if (!$this->definitions) {
            $this->definitions = array_map(
                static fn ($element) => new RouteDefinition($element),
                $this->typeConfiguration
            );
        }

        return $this->definitions;
    }


    private function language(): ?string
    {
        return $this->pageFrameContext->getDimensions()['language'][0] ?? null;
    }
}
