<?php
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\PageFrameProvider\Definition\RoutingConfigurationHelper;
use Newland\PageFrameProvider\Service\PageFrameContextService;

class RoutingConfigurationHelperTest extends FunctionalTestCase
{

    /** @var RoutingConfigurationHelper */
    protected $subject;

    /** @var PageFrameContextService */
    protected $context;

    public function setUp(): void
    {
        parent::setUp();

        $this->context = new PageFrameContextService();
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->subject = new RoutingConfigurationHelper();
        $this->inject($this->subject, 'pageFrameContext', $this->context);
        $this->subject->setTypeConfiguration(
            [
                'test-definition' => [
                    'uriSegment' => [
                        'de' => 'german/uri',
                        'en' => '/with/leading/and/trailing/slashes/',
                        '_fallback' => 'fallback/uri',
                    ],
                ],
                'test-definition-without-fallback' => [
                    'uriSegment' => [
                        'de' => 'german/uri',
                    ],
                ],
            ]
        );
    }

    public function testExtractsUriSegmentForType(): void
    {
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->assertEquals(
            'german/uri',
            $this->subject->urlFragmentForType('test-definition')
        );
    }

    public function testReturnsNullIfTypeUnknown(): void
    {
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->assertEquals(
            null,
            $this->subject->urlFragmentForType('unknown-type')
        );
    }

    public function testUsesFallbackDefinitionIfLanguageUnknown(): void
    {
        $this->context->setDimensions([ 'language' => [ 'ru' ] ]);

        $this->assertEquals(
            'fallback/uri',
            $this->subject->urlFragmentForType('test-definition')
        );
    }

    public function testReturnsNullIfTypeDefinedButLanguageUnknownAndNoFallbackLanguageDefined(): void
    {
        $this->context->setDimensions([ 'language' => [ 'ru' ] ]);

        $this->assertEquals(
            null,
            $this->subject->urlFragmentForType('test-definition-without-fallback')
        );
    }

    public function testStripsLeadingAndTrailingSlashesFromUri(): void
    {
        $this->context->setDimensions([ 'language' => [ 'en' ] ]);

        $this->assertEquals(
            'with/leading/and/trailing/slashes',
            $this->subject->urlFragmentForType('test-definition')
        );
    }

    public function testRemovesUriSegmentsFromRoutePath(): void
    {
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->assertEquals(
            '/foobar',
            $this->subject->removeUriSegmentsFromRoutePath('german/uri/foobar')
        );
    }

    public function testRemovesUriSegmentsFromRoutePathsThatAreDefinedWithLeadingAndTrailingSlashes(): void
    {
        $this->context->setDimensions([ 'language' => [ 'en' ] ]);

        $this->assertEquals(
            '/foobar',
            $this->subject->removeUriSegmentsFromRoutePath('with/leading/and/trailing/slashes/foobar')
        );
    }

    public function testDefaultsAndLanguageUriSegmentsDoNotConflictWithOneAnother(): void
    {
        // Fallback `event` is a substring of `events`. If replacement is done in wrong
        // order, then it'll match `event` first and completely ignore the longer, more
        // precise string in the process.
        $this->subject->setTypeConfiguration([
            '_fallback' => 'event',
            'test-definition' => [
                'uriSegment' => [ 'en' => 'events' ],
            ],
        ]);
        $this->context->setDimensions([ 'language' => [ 'en' ] ]);

        $result = $this->subject->removeUriSegmentsFromRoutePath('events/foo/bar');
        $this->assertNotEquals('s/foo/bar', $result);
        $this->assertEquals('/foo/bar', $result);
    }
}
