<?php

use Neos\Flow\Mvc\Routing\Dto\RouteParameters;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Neos\Domain\Service\ConfigurationContentDimensionPresetSource;
use Newland\PageFrameProvider\Routing\FrontendNodeRoutePartHandler;
use Newland\PageFrameProvider\Routing\RoutingHelper;

class RoutingHelperTest extends FunctionalTestCase
{

    /** @var RoutingHelper */
    protected $subject;

    /** @var ConfigurationContentDimensionPresetSource */
    protected $dimensionSource;

    protected $dimensionConfiguration = [
        'language' => [
            'default' => 'de',
            'defaultPreset' => 'de',
            'label' => 'Language',
            'icon' => 'icon-language',
            'presets' => [
                'de' => [
                    'label' => 'German',
                    'values' => [ 'de_DE' ],
                    'uriSegment' => 'de',
                ],
                'en' => [
                    'label' => 'English',
                    'values' => [ 'en_US' ],
                    'uriSegment' => 'en',
                ],
            ],
        ],
    ];

    public function setUp()
    {
        parent::setUp();
        $this->subject = new RoutingHelper();

        $this->dimensionSource = new ConfigurationContentDimensionPresetSource();
        $this->dimensionSource->setConfiguration($this->dimensionConfiguration);
        $this->inject($this->subject, 'contentDimensionPresetSource', $this->dimensionSource);
    }

    public function testPassesEmptyStringToMatcherIfNoDimensionsInUrl()
    {
        $this->assertSame(
            '',
            $this->subject->extractDimensionValuesFromPath('foo/bar/baz')
        );
    }

    public function testPassesDimensionsToMatcherIfAvailable()
    {
        $this->assertSame(
            'de',
            $this->subject->extractDimensionValuesFromPath('de/foo/bar/baz')
        );
        $this->assertSame(
            'en',
            $this->subject->extractDimensionValuesFromPath('en/foo/bar/baz')
        );
    }

    public function testHandlesMultipleDimensionsInUrlCorreclty()
    {
        $this->dimensionSource->setConfiguration(
            [
                'language' => $this->dimensionConfiguration['language'],
                'somethingElse' => [
                    'name' => 'Foo',
                    'default' => 'foo',
                    'defaultPreset' => 'foo',
                    'presets' => [
                        'foo' => [
                            'label' => 'Foo',
                            'values' => [ 'foo' ],
                            'uriSegment' => 'foo',
                        ],
                        'bar' => [
                            'label' => 'Bar',
                            'values' => [ 'bar' ],
                            'uriSegment' => 'bar',
                        ],
                    ],
                ],
            ]
        );

        $this->assertSame(
            'de_foo',
            $this->subject->extractDimensionValuesFromPath('de_foo/path/goes/here')
        );
        $this->assertSame(
            'en_bar',
            $this->subject->extractDimensionValuesFromPath('en_bar/path/goes/here')
        );
    }

    public function testStripsDimensionsFromUrlIfAvailable()
    {
        $this->assertNotContains(
            'de',
            $this->subject->prepareUrlForFurtherUseInRoutingPipeline('de/foo/bar/baz')
        );
    }

    public function testAddsLeadingSlashIfDimensionsInUrl()
    {
        $this->assertStringStartsWith(
            '/',
            $this->subject->prepareUrlForFurtherUseInRoutingPipeline('de/foo/bar/baz')
        );
    }

    public function testAddsLeadingSlashIfDimensionsNotInUrl()
    {
        $this->assertStringStartsWith(
            '/',
            $this->subject->prepareUrlForFurtherUseInRoutingPipeline('foo/bar/baz')
        );
    }

    public function testDoesNoStripPartsFromUrlThatAreSimilarToDefinedDimensions()
    {
        $this->assertSame(
            '/en/de',
            $this->subject->prepareUrlForFurtherUseInRoutingPipeline('de/en/de')
        );
    }

}