<?php declare(strict_types=1);

namespace Newland\NeosViewHelpers\ViewHelpers\Collection;

use Neos\Flow\Exception;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractViewHelper;

/**
 * Fetches the given key from the given collection.
 * This viewhelper will return the value at the given key in the collection passed or `null`
 * if that key does not exist.
 *
 * A collection can be passed in 2 ways:
 * - Using the `collection` attribute
 * - Using pipes / children
 *
 * WARNING:
 * This ViewHelper is meant to be used in one of the 2 ways mentioned in the examples. It is not recommended
 * to pass a collection as a child in HTML-/XML- Notation as string conversion being done there might transform
 * the collection into a different format.
 *
 * @example
 * <vh:collection.get collection="{myCollection}" key="foo" />
 *
 * @example
 * {myCollection -> vh:collection.get(key: 'foo')}
 */
class GetViewHelper extends AbstractViewHelper
{

    public function initializeArguments(): void
    {
        $this->registerArgument('key', 'string', 'Key to extract', true);
        $this->registerArgument(
            'collection',
            'array',
            'Collection to extract from. If this is not supplied, then the children context will be evaluated',
            false,
            null
        );
    }

    public static function renderStatic(
        array $arguments,
        \Closure $renderChildrenClosure,
        RenderingContextInterface $renderingContext
    ) {
        $collection = $arguments['collection'] ?? static::getCollectionFromChildren($renderChildrenClosure);
        $key = $arguments['key'];

        if (!is_array($collection)) {
            static::throwInvalidTypeException($collection);
        }

        return $collection[$key] ?? null;
    }

    private static function getCollectionFromChildren(\Closure $renderChildrenClosure)
    {
        try {
            return $renderChildrenClosure();
        } catch (Exception $e) {
            if (str_contains($e->getMessage(), 'to string conversion')) {
                throw new Exception(
                    'Error evaluating viewhelper children as array.
                    If you are using XML- / HTML- Notation and passing the collection as a child then this might be
                    due to fluid rendering the children as a string instead of passing it directly.
                    To be sure of the collection type, please use the inline notation of this viewhelper:
                    {myCollection -> vh:collection.get(key: "foo")}',
                    $e->getCode(),
                    $e
                );
            }

            throw $e;
        }
    }

    private static function throwInvalidTypeException($collection): void
    {
        if (empty($collection)) {
            throw new \InvalidArgumentException('
                No collection passed. vh:collection.get must receive a collection using either the `collection`
                attribute or pipes / children evaluation. Any of the following formats should work:
                - {myCollection -> vh:collection.get(key: "foo")}
                - <vh:collection.get collection="{myCollection}" key="foo" />
            ');
        }

        if (is_string($collection)) {
            throw new \InvalidArgumentException('
                Collection passed to vh:collection.get must be of type array, string passed.
                If you are using XML- / HTML- Notation and passing the collection as a child then this might be
                due to fluid rendering the children as a string instead of passing it directly.
                To be sure of the collection type, please use the inline notation of this viewhelper:
                {myCollection -> vh:collection.get(key: "foo")}
            ');
        }

        throw new \InvalidArgumentException(sprintf(
            'Collection passed to vh:collection.get must be of type array, %s passed.',
            gettype($collection)
        ));
    }
}
